#ifndef NFSCSAVELIB_H
#define NFSCSAVELIB_H

#ifdef NFSCSAVELIB_EXPORTS
#define NFSCSLAPI __declspec(dllexport)
#else
#define NFSCSLAPI __declspec(dllimport)
#endif

#define MAGIC_COMPBLK   0x55441122
#define MAGIC_JDLZ      'ZLDJ'
#define MAGIC_HUFF      'FFUH'

#pragma pack(push, 1)

typedef struct tagCompressedFileHeader
{
	unsigned long    Magic;           // defines the compression algorithm, can be JDLZ ('ZLDJ') or HUFF ('FFUH')
	unsigned char    Type;            // not sure, must be 2 for JDLZ and 1 for HUFF
	unsigned char    SizeOfHeader;    // not sure, must be 16
	unsigned short   Unk1;            // zero
	unsigned long    UnpDataSize;     // unpacked data size
	unsigned long    PakDataSize;     // packed data size
} CompressedFileHeader;

typedef struct tagCompressedBlock
{
	unsigned long    Magic;           // 0x55441122
	unsigned long    UnpDataSize;     // must be equal to CompressedFileHeader.UnpDataSize
	unsigned long    SizeOfBlock;     // must be equal to (sizeof(CompressedBlock)-sizeof(CompressedFileHeader)+CompressedFileHeader.PakDataSize) aligned on 4 byte boundary
	unsigned char    SomeData[12];    // some unknown values
	CompressedFileHeader CompFileHdr;
} CompressedBlock;

typedef struct tagSaveHeader
{
	unsigned long    Magic;           // 20CM ('MC02')
	unsigned long    SizeOfSave;      // file size
	unsigned long    ChkSumBlkOffset; // not sure, always 16
	unsigned long    SizeOfData;      // usually includes DataEAHash
	unsigned long    CRC32HdrData;    // crc32b of 16-byte block (SomeData)
	unsigned long    CRC32Data;       // crc32b of save data (beginning from DataEAHash)
	unsigned long    CRC32Hdr;        // crc32b of 24-byte header block (Magic...CRC32Data)
	unsigned char    SomeData[16];    // some unknown values
	unsigned char    DataEAHash[16];  // 16-byte EAHash of data after this header
} SaveHeader;

#pragma pack(pop)

//////////////////////////////////////////////////////////////////////////
// CRC32b - calculates crc32b of data
//
// pData  - pointer to data in memory
// cbData - size of data pointed by pData in bytes
//
// return: unsigned long crc32b value or zero on error
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI unsigned long CRC32b(const void *pData, unsigned long cbData);

//////////////////////////////////////////////////////////////////////////
// MD5 - calculates md5 hash of data
// Part of OpenSSL sources - Copyright (C) 1995-1998 Eric Young (eay@cryptsoft.com)
//
// pData  - pointer to data in memory
// cbData - size of data pointed by pData in bytes
// pMD5   - pointer to the 16-byte buffer to store md5 hash value in
//
// return: true if passed parameter validation (hash value is written to
// pMD5 upon successfull completion) or false otherwise
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI bool MD5(const void *pData, unsigned long cbData, void *pMD5);

//////////////////////////////////////////////////////////////////////////
// EAHash - calculates 128-bit (MD5/custom) hash of data (ripped from NFSC.exe)
//
// pData   - pointer to data in memory
// cbData  - size of data pointed by pData in bytes
// pEAHash - pointer to the 16-byte buffer to store EA hash value in
//
// return: true if passed parameter validation (hash value is written to
// pEAHash upon successfull completion) or false otherwise
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI bool EAHash(const void *pData, unsigned long cbData, void *pEAHash);

//////////////////////////////////////////////////////////////////////////
// JDLZ_compress - compress data using LZ algorithm (ripped from NFSC.exe)
//
// pDataIn  - pointer to data in memory to compress
// cbData   - size of data pointed by pDataIn in bytes
// pDataOut - pointer to the buffer that will recieve compressed data (as JDLZ file)
//
// return: unsigned long size in bytes of compressed data or zero on error
//
// Note: size of pDataOut is recommended to be at least cbData+sizeof(JDLZHeader)
// or even more; in most cases that may be a waste of memory, but that will be
// the price for stability (avoiding access violations)
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI unsigned long JDLZ_compress(const void *pDataIn, unsigned long cbData, void *pDataOut);

//////////////////////////////////////////////////////////////////////////
// JDLZ_decompress - decompress data in memory using LZ algorithm (ripped from NFSC.exe)
//
// pDataIn  - pointer to the JDLZ file in memory
// pDataOut - pointer to the buffer that will recieve uncompressed data
//
// return: unsigned long size in bytes of uncompressed data or zero on error
//
// Note: size of pDataOut can be obtained from the JDLZ file header (refer to the
// JDLZHeader structure) in case you have a valid JDLZ file/data block in memory
// pointed by pDataIn; the return value id usually equal to UnpDataSize
// field of JDLZ file header
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI unsigned long JDLZ_decompress(const void *pDataIn, void *pDataOut);

//////////////////////////////////////////////////////////////////////////
// IsSave - check if pSaveData points to a valid save header (beginning of save file)
//
// pSaveData - pointer to save file in memory
// cbData    - size of data pointed by pSaveData in bytes
//
// return: true if all checks were passed or false otherwise
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI bool IsSave(const void *pSaveData, unsigned long cbData);

//////////////////////////////////////////////////////////////////////////
// SaveIsChkSumValid - perform save file checksum validation (exactly as the game does)
//
// pSaveData - pointer to save file in memory
// cbData    - size of data pointed by pSaveData in bytes
//
// return: true if all checks were passed or false otherwise
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI bool SaveIsChkSumValid(const void *pSaveData, unsigned long cbData);

//////////////////////////////////////////////////////////////////////////
// SaveFixChkSum - fix save file checksums
//
// pSaveData - pointer to save file in memory (must have read+write access)
// cbData    - size of data pointed by pSaveData in bytes
//
// return: true if the save file was successfully updated or false otherwise
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI bool SaveFixChkSum(void *pSaveData, unsigned long cbData);

//////////////////////////////////////////////////////////////////////////
// SaveRepair - make save file valid for this computer
//
// pSaveData - pointer to save file in memory (must have read+write access)
// cbData    - size of data pointed by pSaveData in bytes
//
// return: true if the save file was successfully repaired or false otherwise
//
// Note: the save is considered valid on the given computer if it's checksums are
// correct and if the game serial number stored in the save file is equal to the
// one saved in registry of the given computer, but if there is no serial stored
// in registry, the save file must not also contain something except zeroes in the
// serial number field; so, the function updates the serial in the save file to match
// one stored in registry (or zeroes it if it is missing) and updates the checksums
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI bool SaveRepair(void *pSaveData, unsigned long cbData);

//////////////////////////////////////////////////////////////////////////
// ReadCDKeyFromRegistry - read NFSC CD Key stored in registry
//
// pszCDKey - pointer to the buffer that will receive zero-terminated CD Key string
// cbCDKey  - size of buffer pointed by pszCDKey in bytes
//
// return: true if the CD Key was stored in pszCDKey or false otherwise
//
// Note: even if the result is true, the pszCDKey variable can point to an empty
// string in case that CD Key is not stored in registry or cannot be read due to
// some error, so false result is returned only if the buffer supplied is not
// large enough to hold the CD Key value
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI bool ReadCDKeyFromRegistry(char *pszCDKey, unsigned long cbCDKey);

//////////////////////////////////////////////////////////////////////////
// WriteCDKeyToRegistry - write NFSC CD Key to registry
//
// pszCDKey - pointer to zero-terminated CD Key string
//
// return: true if the CD Key was successfully stored in registry or false otherwise
//
// Note: the string pointed by pszCDKey must not contain '-' characters and
// must have the length of 20 chars
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI bool WriteCDKeyToRegistry(const char *pszCDKey);

//////////////////////////////////////////////////////////////////////////
// DeleteCDKeyFromRegistry - deletes NFSC CD Key from registry
//
// return: true if the CD Key was successfully removed or false on error
//////////////////////////////////////////////////////////////////////////

NFSCSLAPI bool DeleteCDKeyFromRegistry();

#endif