/*
    Copyright 2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

//#define NOLFS
#ifndef NOLFS   // 64 bit file support not really needed since the tool uses signed 32 bits at the moment, anyway I leave it enabled
    #define _LARGE_FILES        // if it's not supported the tool will work
    #define __USE_LARGEFILE64   // without support for large files
    #define __USE_FILE_OFFSET64
    #define _LARGEFILE_SOURCE
    #define _LARGEFILE64_SOURCE
    #define _FILE_OFFSET_BITS   64
#endif

// gcc -o quickbms -lz -llzo -lbz2 src/*.c
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>
#include <errno.h>

//typedef int8_t      i8;
typedef uint8_t     u8;
//typedef int16_t     i16;
typedef uint16_t    u16;
//typedef int32_t     i32;
//typedef uint32_t    u32;

#include <zlib.h>
#include <bzlib.h>
#include "LzmaDec.h"
#include "Lzma2Dec.h"
#include "Bra.h"
#include "blast.h"
#ifndef DISABLE_LZO     // add -DDISABLE_LZO at compiling if you don't have LZO
    #include <lzo/lzo1.h>
    #include <lzo/lzo1a.h>
    #include <lzo/lzo1b.h>
    #include <lzo/lzo1c.h>
    #include <lzo/lzo1f.h>
    #include <lzo/lzo1x.h>
    #include <lzo/lzo1y.h>
    #include <lzo/lzo1z.h>
    #include <lzo/lzo2a.h>
#endif

#include "aes.h"    // most of the encryption function come from http://polarssl.org
#include "blowfish.h"
#include "des.h"
#include "arc4.h"
#include "tea.h"
#include "xtea.h"
#include "idea.h"
#include "myenc.h"
void xtea_crypt_ecb( xtea_context *ctx, int mode, unsigned char input[8], unsigned char output[8] );

#ifdef __DJGPP__
    #define NOLFS
    char **__crt0_glob_function (char *arg) { return 0; }
    void   __crt0_load_environment_file (char *progname) { }
#endif

#ifdef WIN32
    #include <windows.h>
    #include <direct.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)

    HWND    mywnd   = NULL;
    char *get_file(char *title, int bms);
    char *get_folder(char *title);
#else
    #include <dirent.h>

    #define stricmp     strcasecmp
    #define strnicmp    strncasecmp
    #define stristr     strcasestr
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
#endif

#if defined(_LARGE_FILES)
    #if defined(__APPLE__)
        #define fseek   fseeko
        #define ftell   ftello
    #elif defined(__FreeBSD__)
    #elif !defined(NOLFS)       // use -DNOLFS if this tool can't be compiled on your OS!
        #define off_t   off64_t
        #define fopen   fopen64
        #define fseek   fseeko64
        #define ftell   ftello64
    #endif
#endif



#define VER             "0.3.1"
#define BUFFSZ          8192
#define MAX_ARGS        16      // fixed but exagerated
#define MAX_VARS        1024    // fixed but exagerated
#define MAX_FILES       512     // fixed but exagerated
#define MAX_CMDS        1024    // fixed but exagerated
#define MAX_ARRAYS      128     // fixed but exagerated

#define STRINGSZ        256
#define NUMBERSZ        24      // ready for 64 bits, includes also space for the NULL delimiter
#define PATHSZ          4096

#define MYLITTLE_ENDIAN 0
#define MYBIG_ENDIAN    1

#define INTSZ           32
#define int             int32_t     // trick for forcing the usage of signed 32 bit numbers on any system without modifying the code
#define u_int           uint32_t    // used only in some rare occasions

#define CMD             command[cmd]
#define ARG             argument
#define NUM(X)          CMD.num[X]
#define STR(X)          CMD.str[X]
#define VARISNUM(X)     variable[CMD.var[X]].isnum
#define VAR(X)          get_var(CMD.var[X])
#define VAR32(X)        get_var32(CMD.var[X])
#define VARSZ(X)        variable[CMD.var[X]].size   // due to the memory enhancement done on this tool, VARSZ returns ever STRINGSZ for sizes lower than this value... so do NOT trust this value!
//#define FILEZ(X)        ((NUM(X) < 0) ? NULL : filenumber[NUM(X)].fd)  // automatic support for MEMORY_FILE
#define FILEZ(X)        NUM(X)
#define MEMORY_FNAME    "MEMORY_FILE"
#define MEMORY_FNAMESZ  (sizeof(MEMORY_FNAME) - 1)
#define TEMPORARY_FILE  "TEMPORARY_FILE"
#define ALLOC_ERR       alloc_err(__FILE__, __LINE__, __FUNCTION__)
#define STD_ERR         std_err(__FILE__, __LINE__, __FUNCTION__)
#define myatoi(X)       readbase(X, 10, NULL)
#define CSTRING(X,Y)    CMD.str[X] = strdup(Y); \
                        CMD.num[X] = cstring(CMD.str[X], CMD.str[X], -1, NULL);
#define NUMS2BYTES(A,B,C,D) \
                        tmp = numbers_to_bytes(A, &B); \
                        myalloc(&C, B, &D); \
                        memcpy(C, tmp, B);
#define FREEX(X,Y)      if(X) { \
                            Y; \
                            free(X); \
                            X = NULL; \
                        }
#define strdup_error    "Error: do NOT use strdup, use re_strdup!"
#define strdup          strdup_error



enum {
    CMD_NONE = 0,
    CMD_CLog,
    CMD_Do,
    CMD_FindLoc,
    CMD_For,
    CMD_ForTo,  // for an easy handling of For
    CMD_Get,
    CMD_GetDString,
    CMD_GoTo,
    CMD_IDString,
    CMD_ImpType,
    CMD_Log,
    CMD_Math,
    CMD_Next,
    CMD_Open,
    CMD_SavePos,
    CMD_Set,
    CMD_While,
    CMD_String,
    CMD_CleanExit,
    CMD_If,
    CMD_Else,
    CMD_Elif,   // added by me
    CMD_EndIf,
    CMD_GetCT,
    CMD_ComType,
    CMD_ReverseLong,
        // added by me
    CMD_Endian,
    CMD_FileXOR,
    CMD_FileRot13,
    CMD_Break,
    CMD_Strlen,
    CMD_GetVarChr,
    CMD_PutVarChr,
    CMD_Debug,
    CMD_Padding,
    CMD_Append,
    CMD_Encryption,
    CMD_Print,
    CMD_GetArray,
    CMD_PutArray,
    CMD_StartFunction,
    CMD_CallFunction,
    CMD_EndFunction,
    CMD_ScanDir,
        // nop
    CMD_NOP
};

#define ISNUMTYPE(X)    ((X > 0) || (X == TYPE_ASIZE))
enum {  // the value is referred to their size which makes the job faster, numbers are positive and the others are negative!
    TYPE_NONE           = 0,
    TYPE_BYTE           = 1,
    TYPE_INT            = 2,
    TYPE_THREEBYTE      = 3,
    TYPE_LONG           = 4,
    TYPE_STRING         = -1,
    TYPE_ASIZE          = -2,
    TYPE_PURETEXT       = -3,
    TYPE_PURENUMBER     = -4,
    TYPE_TEXTORNUMBER   = -5,
    TYPE_FILENUMBER     = -6,
        // added by me
    TYPE_FILENAME       = -1000,
    TYPE_BASENAME       = -1001,
    TYPE_EXTENSION      = -1002,
    TYPE_UNICODE        = -1003,
    TYPE_BINARY         = -1004,
    TYPE_LINE           = -1005,
};

enum {
    COMP_NONE = 0,
    COMP_ZLIB,      // RFC 1950
    COMP_DEFLATE,   // RFC 1951
    COMP_LZO1,
    COMP_LZO1A,
    COMP_LZO1B,
    COMP_LZO1C,
    COMP_LZO1F,
    COMP_LZO1X,
    COMP_LZO1Y,
    COMP_LZO1Z,
    COMP_LZO2A,
    COMP_LZSS,
    COMP_LZX,
    COMP_GZIP,
    COMP_EXPLODE,
    COMP_LZMA,
    COMP_LZMA_86HEAD,
    COMP_LZMA_86DEC,
    COMP_LZMA_86DECHEAD,
    COMP_BZ2,
    COMP_XMEMLZX,
    COMP_HEX,
    COMP_BASE64,
    COMP_LZW,
    COMP_LZWX,
    COMP_LZSSBOH,
    //COMP_CAB,
    //COMP_CHM,
    //COMP_SZDD
    COMP_LZXCAB,
    COMP_LZXCHM,
    COMP_RLEW,
    COMP_LZJB,
    COMP_SFL_BLOCK,
    COMP_SFL_RLE,
    COMP_SFL_NULLS,
    COMP_SFL_BITS,
    COMP_LZMA2,
    COMP_LZMA2_86HEAD,
    COMP_LZMA2_86DEC,
    COMP_LZMA2_86DECHEAD
};

enum {
    LZMA_FLAGS_NONE         = 0,
    LZMA_FLAGS_86_HEADER    = 1,
    LZMA_FLAGS_86_DECODER   = 2
};

typedef struct {
    u8      *name;          // name of the variable, it can be also a fixed number since "everything" is handled as a variable
    u8      *value;         // it's current value in the form of an allocated string
    int     value32;        // number
    int     isnum;          // 1 if it's a number, 0 if a string
    u8      constant;       // it's 1 if the variable is a fixed number and not a "real" variable
    int     size;           // used for avoiding to waste realloc too much, not so much important and well used in reality
} variable_t;

typedef struct {
    u8      type;           // type of command to execute
    u8      *debug_line;    // used with -v
    int     var[MAX_ARGS];  // pointer to a variable
    int     num[MAX_ARGS];  // simple number
    u8      *str[MAX_ARGS]; // fixed string
} command_t;

typedef struct {
    FILE    *fd;
    u8      *fullname;      // just the same input filename, like c:\myfile.pak or ..\..\myfile.pak
    u8      *filename;      // input filename only, like myfile.pak
    u8      *basename;      // input basename only, like myfile
    u8      *fileext;       // input extension only, like pak
} filenumber_t;

typedef struct {
    u8      *data;
    int     pos;
    int     size;
    int     maxsize;
} memory_file_t;

typedef struct {
    int     elements;
    u8      **str;
} array_t;

typedef struct {
    u8      *name;
    int     offset; // unused at the moment
    int     size;
} files_t;

filenumber_t    filenumber[MAX_FILES + 1];
variable_t      variable_main[MAX_VARS + 1];
variable_t      *variable = variable_main;
command_t       command[MAX_CMDS + 1];
memory_file_t   memory_file[MAX_FILES + 1];
array_t         array[MAX_ARRAYS + 1];

aes_context  *aes_ctx       = NULL;
blf_ctx      *blowfish_ctx  = NULL;
des_context  *des_ctx       = NULL;
des3_context *des3_ctx      = NULL;
arc4_context *rc4_ctx       = NULL;
tea_context  *tea_ctx       = NULL;
xtea_context *xtea_ctx      = NULL;
IDEA_context *idea_ctx      = NULL;
xor_context  *xor_ctx       = NULL;
rot_context  *rot_ctx       = NULL;
charset_context *charset_ctx = NULL;
FILE    *listfd             = NULL;
int     bms_line_number     = 0,
        extracted_files     = 0,
        endian              = MYLITTLE_ENDIAN,
        list_only           = 0,
        force_overwrite     = 0,
        verbose             = 0,
        variables           = 0,
        quick_gui_exit      = 0,
        compression_type    = COMP_ZLIB,
        *file_xor_pos       = NULL,
        file_xor_size       = 0,
        *file_rot13_pos     = NULL,
        file_rot13_size     = 0,
        append_mode         = 0,
        quickbms_version    = 0,
        decimal_notation    = 1,        // myitoa is a bit slower (due to the %/) but is better for some strings+num combinations
        mex_default         = 0;
        //min_int             = 1 << ((sizeof(int) << 3) - 1),
        //max_int             = (u_int)(1 << ((sizeof(int) << 3) - 1)) - 1;
u8      input_folder[PATHSZ + 1] = "",  // just the current folder when the program is launched
        *filter_files       = NULL,     // the wildcard
        *filter_in_files    = NULL,     // the wildcard
        *file_xor           = NULL,     // contains all the XOR numbers
        *file_rot13         = NULL,     // contains all the rot13 numbers
        *encryption_ivec    = NULL;
int     EXTRCNT_idx         = 0,
        BytesRead_idx       = 0,
        NotEOF_idx          = 0;



int check_is_dir(u8 *fname);
int readbase(u8 *data, int size, int *readn);
void myfopen(u8 *fname, int fdnum);
void mex_default_init(int file_only);
void bms_init(int reinit);
void bms_finish(void);
files_t *add_files(u8 *fname, int fsize, int *ret_files);
int recursive_dir(u8 *filedir);
int start_bms(int startcmd, int nop, int *ret_break);
int parse_bms(FILE *fds);
int bms_line(FILE *fd, u8 **argument, u8 **debug_line);
int cstring(u8 *input, u8 *output, int maxchars, int *inlen);
int myisalnum(int chr);
int myisdigitstr(u8 *str);
int myisdigit(int chr);
u8 *myitoa(int num);
int myatoifile(u8 *str);
//int myatoi(u8 *str);
u8 *re_strdup(u8 *dst, u8 *src, int *retlen);
int add_var(int idx, u8 *str, u8 *val, int val32, int valsz);
void dumpa_memory_file(memory_file_t *memfile, u8 *data, int size);
int dumpa(int fdnum, u8 *fname, int offset, int size, int zsize);
int hex2byte(u8 *hex);
int check_wildcard(u8 *fname, u8 *wildcard);
u8 *create_dir(u8 *name);
int check_overwrite(u8 *fname);
void myalloc(u8 **data, int wantsize, int *currsize);
int getxx(u8 *tmp, int bytes);
int putxx(u8 *data, int num, int bytes);
u8 *fgetss(int fdnum, int chr, int unicode, int line);
int myfgetc(int fdnum);
int fgetxx(int fdnum, int bytes);
u8 *myfrx(int fdnum, int type, int *ret_num, int *error);
void post_fseek_actions(int fdnum, int diff_offset);
void post_fread_actions(int fdnum, u8 *data, int size);
int myftell(int fdnum);
int myfeof(int fdnum);
void myfseek(int fdnum, int offset, int type);
int myfr(int fdnum, u8 *data, int size);
void myhelp(u8 *arg0);
void quick_bms_list(void);
int calc_quickbms_version(u8 *version);
void alloc_err(const char *fname, int line, const char *func);
void std_err(const char *fname, int line, const char *func);
void myexit(int ret);
#include "unlz.h"



int main(int argc, char *argv[]) {
    files_t *files      = NULL;
    FILE    *fds;
    int     i,
            mybreak     = 0,
            curr_file   = 0,
            total_files = 0;
    u8      filedir[PATHSZ + 1] = ".",
            bckdir[PATHSZ + 1]  = ".",
            *newdir     = NULL,
            *bms,
            *fname,
            *fdir,
            *listfile   = NULL;

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n"
        "QuickBMS generic files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

#ifdef WIN32
    mywnd = GetForegroundWindow();
    if(GetWindowLong(mywnd, GWL_WNDPROC)) {
        for(i = 1; i < argc; i++) {
            if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
                break;
            }
        }
        i = 3 - (argc - i);
        if(i > 0) {
            bms = calloc(sizeof(char *), argc + i + 1);
            if(!bms) STD_ERR;
            memcpy(bms, argv, sizeof(char *) * argc);
            argv = (void *)bms;
            argc -= (3 - i);
            if(i >= 3) argv[argc]     = get_file("select the BMS script to use", 1);
            if(i >= 2) argv[argc + 1] = get_file("select the input archive/file to extract", 0);
            if(i >= 1) argv[argc + 2] = get_folder("select the output folder where extracting the files");
            argc += 3;
        }
    }
#endif

    if(argc < 4) {
        if((argc >= 2) && (argv[1][1] == 'c')) {
            quick_bms_list();
            myexit(-1);
        }
        myhelp(argv[0]);
        myexit(-1);
    }

    argc -= 3;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            myexit(-1);
        }
        switch(argv[i][1]) {
            case '-':
            case '?':
            case 'h': { myhelp(argv[0]);  myexit(-1); } break;
            case 'c': { quick_bms_list(); myexit(-1); } break;
            case 'l': list_only         = 1;            break;
            case 'f': filter_files      = argv[++i];    break;
            case 'F': filter_in_files   = argv[++i];    break;
            case 'o': force_overwrite   = 1;            break;
            case 'v': verbose           = 1;            break;
            case 'L': listfile          = argv[++i];    break;
            case 'R': quick_gui_exit    = 1;            break;  // internal usage for external programs
            case 'x': decimal_notation  = 0;            break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                myexit(-1);
            }
        }
    }

    bms   = argv[argc];
    fname = argv[argc + 1];
    fdir  = argv[argc + 2];

    bms_init(0);

    if(check_is_dir(fname)) {
        newdir = fname;
        printf("- start the scanning of the input folder: %s\n", newdir);
        getcwd(bckdir, sizeof(bckdir));
        if(chdir(newdir) < 0) STD_ERR;
        strcpy(filedir, ".");
        recursive_dir(filedir);
        files = add_files(NULL, 0, &total_files);
        curr_file = 0;
        if(total_files <= 0) {
            printf("\nError: the input folder is empty\n");
            myexit(-1);
        }
        chdir(bckdir);
    }

redo:
    if(files) {
        fname = files[curr_file].name;
        curr_file++;
        chdir(newdir);
    }
    myfopen(fname, 0);
    if(files) {
        chdir(bckdir);
    }

    printf("- open script %s\n", bms);
    fds = fopen(bms, "rb");
    if(!fds) STD_ERR;
    parse_bms(fds);
    fclose(fds);

    if(listfile) {
        listfd = fopen(listfile, "wb");
        if(!listfd) STD_ERR;
    }

    if(!list_only) {
        printf("- set output folder %s\n", fdir);
        if(chdir(fdir) < 0) STD_ERR;
    }

    printf("\n"
        "  offset   filesize   filename\n"
        "------------------------------\n");
    start_bms(-1, 0, &mybreak);

    printf("\n- %d files found\n", extracted_files);

    if(files && (curr_file < total_files)) {
        bms_init(1);
        goto redo;
    }

    bms_finish();
    if(listfile) {
        fclose(listfd);
    }
    myexit(0);
    return(0);
}



int check_is_dir(u8 *fname) {
    struct stat xstat;

    if(stat(fname, &xstat) < 0) return(0);
    if(!S_ISDIR(xstat.st_mode)) return(0);
    return(1);
}



#ifdef WIN32
char *get_file(char *title, int bms) {
    OPENFILENAME    ofn;
    char    *filename;

    filename = malloc(PATHSZ + 1);
    if(!filename) STD_ERR;
    filename[0] = 0;
    memset(&ofn, 0, sizeof(ofn));
    ofn.lStructSize     = sizeof(ofn);
    if(bms) {
        ofn.lpstrFilter =
            "BMS script\0"  "*.bms;*.txt\0"
            "(*.*)\0"       "*.*\0"
            "\0"            "\0";
    } else {
        ofn.lpstrFilter =
            "(*.*)\0"       "*.*\0"
            "\0"            "\0";
    }
    ofn.nFilterIndex    = 1;
    ofn.lpstrFile       = filename;
    ofn.nMaxFile        = PATHSZ;
    ofn.lpstrTitle      = title;
    ofn.Flags           = OFN_PATHMUSTEXIST | OFN_LONGNAMES | OFN_EXPLORER | OFN_HIDEREADONLY | OFN_FILEMUSTEXIST;

    printf("- %s\n", ofn.lpstrTitle);
    if(!GetOpenFileName(&ofn)) exit(1); // terminate immediately
    return(filename);
}

char *get_folder(char *title) {
    OPENFILENAME    ofn;
    char    *p;
    char    *filename;

    filename = malloc(PATHSZ + 1);
    if(!filename) STD_ERR;

    strcpy(filename, "enter in the output folder and press Save");
    memset(&ofn, 0, sizeof(ofn));
    ofn.lStructSize     = sizeof(ofn);
    ofn.lpstrFilter     = "(*.*)\0" "*.*\0" "\0" "\0";
    ofn.nFilterIndex    = 1;
    ofn.lpstrFile       = filename;
    ofn.nMaxFile        = PATHSZ;
    ofn.lpstrTitle      = title;
    ofn.Flags           = OFN_PATHMUSTEXIST | OFN_LONGNAMES | OFN_EXPLORER | OFN_HIDEREADONLY;

    printf("- %s\n", ofn.lpstrTitle);
    if(!GetSaveFileName(&ofn)) exit(1); // terminate immediately
    p = strrchr(filename, '\\');
    if(!p) p = strrchr(filename, '/');
    if(p) *p = 0;
    return(filename);
}
#endif



void myfopen(u8 *fname, int fdnum) {
    filenumber_t    *filez;
    u8      *p;

    if((fdnum < 0) || !strnicmp(fname, MEMORY_FNAME, MEMORY_FNAMESZ)) {
        printf("\n"
            "Error: the filenumber field is minor than 0, if you want to use MEMORY_FILE\n"
            "       you don't need to \"reopen\" it in this way, just specify MEMORY_FILE\n"
            "       as filenumber in the various commands like:\n"
            "         get VAR long MEMORY_FILE\n");
        myexit(-1);
    } else if(fdnum >= MAX_FILES) {
        printf("\nError: the BMS script uses more files than how much supported by this tool\n");
        myexit(-1);
    }
    filez = &filenumber[fdnum];

    printf("- open input file %s\n", fname);
    if(filez->fd) fclose(filez->fd);
    filez->fd = fopen(fname, "rb");
    if(!filez->fd) STD_ERR;

    // fullname
    filez->fullname = re_strdup(filez->fullname, fname, NULL);    // allocate

    // filename
    filez->filename = strrchr(filez->fullname, '\\');
    if(!filez->filename) filez->filename = strrchr(filez->fullname, '/');
    if(filez->filename) {
        filez->filename++;
    } else {
        filez->filename = filez->fullname;
    }

    // basename
    filez->basename = re_strdup(filez->basename, filez->filename, NULL);  // allocate
    p = strrchr(filez->basename, '.');
    if(p) *p = 0;

    // extension
    filez->fileext = strrchr(filez->filename, '.');
    if(filez->fileext) {
        filez->fileext++;
    } else {
        filez->fileext = filez->filename + strlen(filez->filename);
    }

    if(mex_default) {
        if(!fdnum) {
            add_var(BytesRead_idx, NULL, NULL, 0, sizeof(int));
            add_var(NotEOF_idx,    NULL, NULL, 1, sizeof(int));
        }
    }
}



int add_datatype(u8 *str) {
    if(!stricmp(str, "Long"))       return(TYPE_LONG);
    if(!stricmp(str, "Int"))        return(TYPE_INT);
    if(!stricmp(str, "Byte"))       return(TYPE_BYTE);
    if(!stricmp(str, "ThreeByte"))  return(TYPE_THREEBYTE);
    if(!stricmp(str, "String"))     return(TYPE_STRING);
    if(!stricmp(str, "ASize"))      return(TYPE_ASIZE);
    // added by me
    if(!stricmp(str, "Short"))      return(TYPE_INT);
    if(!stricmp(str, "Char"))       return(TYPE_BYTE);
    if(!stricmp(str, "4"))          return(TYPE_LONG);
    if(!stricmp(str, "3"))          return(TYPE_THREEBYTE);
    if(!stricmp(str, "2"))          return(TYPE_INT);
    if(!stricmp(str, "1"))          return(TYPE_BYTE);
    if(strstr(str,   "32"))         return(TYPE_LONG);
    if(!stricmp(str, "24"))         return(TYPE_THREEBYTE);
    if(strstr(str,   "16"))         return(TYPE_INT);
    if(strstr(str,   "8"))          return(TYPE_BYTE);
    if(!stricmp(str, "FileName"))   return(TYPE_FILENAME);
    if(!stricmp(str, "BaseName"))   return(TYPE_BASENAME);
    if(!stricmp(str, "Extension"))  return(TYPE_EXTENSION);
    if(!stricmp(str, "FileExt"))    return(TYPE_EXTENSION);
    if(!stricmp(str, "Unicode"))    return(TYPE_UNICODE);
    if(!stricmp(str, "UTF-16"))     return(TYPE_UNICODE);
    if(!stricmp(str, "UTF16"))      return(TYPE_UNICODE);
    if(!stricmp(str, "Binary"))     return(TYPE_BINARY);
    if(!stricmp(str, "Line"))       return(TYPE_LINE);
    printf("\nError: invalid datatype %s at line %d\n", str, bms_line_number);
    myexit(-1);
    return(-1);
}



int get_var_from_name(u8 *name, int namelen) {
    int     i;

    if(namelen < 0) namelen = strlen(name);
    for(i = 0; variable[i].name; i++) {
        if(!strnicmp(variable[i].name, name, namelen) && !variable[i].name[namelen]) return(i);
    }
    return(-1);
}



#define GET_VAR_COMMON(X1,X2,X3) \
    if((idx < 0) || (idx >= MAX_VARS)) { \
        printf("\nError: the variable index is invalid, there is an error in this tool\n"); \
        myexit(-1); \
    } \
    if(variable[idx].isnum) { \
        if(verbose) printf("             >get %s (%d) 0x%08x\n", variable[idx].name, idx, variable[idx].value32); \
        return(X1); \
    } \
    if(variable[idx].value) { \
        if(verbose) printf("             >get %s (%d) %s\n", variable[idx].name, idx, variable[idx].value); \
        return(X2); \
    } \
    if(variable[idx].name[0] && strnicmp(variable[idx].name, MEMORY_FNAME, MEMORY_FNAMESZ)) { /* "" is for sequential file names */ \
        if(verbose) printf("- variable %s seems uninitialized, I use its name\n", variable[idx].name); \
        /* myexit(-1); */ \
    } \
    if(verbose) printf("             >get %s (%d) %s\n", variable[idx].name, idx, variable[idx].name); \
    return(X3);



int get_var32(int idx) {
    GET_VAR_COMMON(
        variable[idx].value32,
        myatoi(variable[idx].value),
        myatoi(variable[idx].name))
}



u8 *get_var(int idx) {
    GET_VAR_COMMON(
        myitoa(variable[idx].value32),
        variable[idx].value,
        variable[idx].name)
}



int readbase(u8 *data, int size, int *readn) {
    static const u8 table[256] =    // fast performances
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\xff\xff\xff\xff\xff\xff"
            "\xff\x0a\x0b\x0c\x0d\x0e\x0f\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\x0a\x0b\x0c\x0d\x0e\x0f\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
            "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff";
    int     num     = 0;
    int     sign;
    u8      c,
            *s;

    s = data;
    if(!data || !size || !data[0]) {
        // do nothing (for readn)
    } else {
        while(*s && (*s <= ' ')) s++;   // useful in some occasions, for example if the input is external!
        if(*s == '-') {
            sign = -1;
            s++;
        } else {
            sign = 0;
        }
        for(; *s; s++) {
            c = *s;
            if((c == 'x') || (c == 'X') || (c == '$')) {  // auto base switching
                size = 16;
                continue;
            }
            c = table[c];
            if(c >= size) break;    // necessary to recognize the invalid chars based on the size
            num = (num * size) + c;
        }
        if(sign) num = -num;
    }
    if(readn) *readn = s - data;
    return(num);
}



u8 *strdupcpy(u8 *dst, int *dstlen, u8 *src, int srclen) {
    if(!dst || (*dstlen < srclen)) {
        *dstlen = srclen;
        if(*dstlen == -1) ALLOC_ERR;
        if(*dstlen < STRINGSZ) *dstlen = STRINGSZ;    // better for numbers and common filenames
        dst = realloc(dst, *dstlen + 1);
        if(!dst) STD_ERR;
    }
    if(dst && src) memcpy(dst, src, srclen);
    if(dst) dst[srclen] = 0;
    return(dst);
}



u8 *re_strdup(u8 *dst, u8 *src, int *retlen) {  // only for NULL delimited strings, NOT bytes!
    int     dstlen  = -1,
            srclen;

    if(retlen) dstlen = *retlen;
    if(src) {
        srclen = strlen(src);
    } else {
        srclen = 0;
    }
    dst = strdupcpy(dst, &dstlen, src, srclen);
    return(dst);
}



void strdup_replace(u8 **dstp, u8 *src, int src_len, int *dstp_len) {  // should improve a bit the performances
    int     dst_len = -1;
    u8      *dst;

    if(!dstp) return;
    dst = *dstp;

    if(!dstp_len && dst) {
        dst_len = strlen(dst);
    } else if(dstp_len) {
        dst_len = *dstp_len;
    }
    if(src_len < 0) {
        if(src) {
            src_len = strlen(src);
        } else {
            src_len = 0;
        }
    }

    dst = strdupcpy(dst, &dst_len, src, src_len);

    *dstp = dst;
    if(dstp_len) *dstp_len = dst_len;
}



int get_memory_file(u8 *str) {
    int     ret = -1;

    // MEMORY_FILE  = -1
    // MEMORY_FILE1 = -1
    // MEMORY_FILE2 = -2

    if(str) {
        ret = myatoi(str + MEMORY_FNAMESZ);
        if(!ret) ret++;
        if(ret > MAX_FILES) {
            printf("\nError: too big MEMORY_FILE number\n");
            myexit(-1);
        }
        ret = -ret;
    }
    if(ret >= 0) {
        printf("\nError: the memory file has a positive number\n");
        myexit(-1);
    }
    return(ret);
}



int add_var(int idx, u8 *str, u8 *val, int val32, int valsz) {
// I have choosed -2 because it's negative and is different than -1, a fast solution
#define ADD_VAL if(valsz != -2) { \
                    if(variable[idx].constant) goto quit_error; \
                    if(val) { \
                        strdup_replace(&variable[idx].value, val, valsz, &variable[idx].size); \
                        variable[idx].isnum   = 0; \
                    } else { \
                        variable[idx].value32 = val32; \
                        variable[idx].isnum   = 1; \
                    } \
                }

    // do NOT touch valsz, it's a job of strdup_replace
    if((idx < 0) || (idx >= MAX_VARS)) {
        printf("\nError: the variable index is invalid, there is an error in this tool\n");
        myexit(-1);
    }
    // if(valsz < 0) valsz = STRINGSZ;  do NOT do this, valsz is calculated on the length of val
    if(!str && (idx >= 0)) {
        str = variable[idx].name;
        ADD_VAL
        goto quit;
    } else {    // used only when the bms file is parsed at the beginning
        if(!stricmp(str, "EXTRCNT") || !stricmp(str, "BytesRead") || !stricmp(str, "NotEOF")) {
            if(!mex_default) {
                mex_default = 1;    // this avoids to waste cpu for these boring variables
                mex_default_init(0);
            }
        }
        for(idx = 0; variable[idx].name; idx++) {
            if(!stricmp(variable[idx].name, str)) {
                ADD_VAL
                goto quit;
            }
        }
        if(idx >= MAX_VARS) {
            printf("\nError: the BMS script uses more variables than how much supported by this tool\n");
            myexit(-1);
        }
        strdup_replace(&variable[idx].name, str, -1, &variable[idx].size);
        ADD_VAL

        // if this "if" is removed the tool will be a bit slower but will be able to handle completely the script in the example below
        //if(myisdigitstr(variable[idx].name)) {  
        if(myisdigit(variable[idx].name[0])) {  // number: why only the first byte? because decimal and hex (0x) start all with a decimal number or a '-'
            //strdup_replace(&variable[idx].value, variable[idx].name, -1, &variable[idx].size);
            variable[idx].value32  = myatoi(variable[idx].name);
            variable[idx].isnum    = 1;
            variable[idx].constant = 1;    // it's like read-only

            // there is only one incompatibility with the string-only variables, but it's acceptable for the moment:
            //   set NAME string "mytest"
            //   set NUM long 0x1234
            //   string NAME += NUM
            //   print "%NAME%"
            //   set NUM string "0x12349999999999"
            //   string NAME += NUM
            //   print "%NAME%"
        }
    }
quit:
    if(verbose) {
        if(variable[idx].isnum) {
            printf("             >set %s (%d) to 0x%08x\n", variable[idx].name, idx, variable[idx].value32);
        } else if(variable[idx].value) {
            printf("             >set %s (%d) to %s\n", variable[idx].name, idx, variable[idx].value);
        } else {
            printf("             >set %s (%d) to %s\n", variable[idx].name, idx, variable[idx].name);
        }
    }
    return(idx);
quit_error:
    printf("\nError: there is something wrong in the BMS, var %d is a constant number\n", idx);
    myexit(-1);
    return(-1);
}



int myisdechex_string(u8 *str) {
    int     len;

    // I have already verified that using a quick test only on the first char doesn't improve the performances if compared to the current full check
    readbase(str, 10, &len);
    if(len <= 0) return(0); // FALSE
    return(1);              // TRUE
}



int check_condition(int cmd) {
    int     var1n   = 0,
            var2n   = 0,
            res;
    u8      *cond,
            *var1   = NULL,
            *var2   = NULL,
            *p;

    if((CMD.var[0] < 0) && (CMD.var[2] < 0)) return(0); // needed for CMD_Else!
    cond = STR(1);
    if(VARISNUM(0) && VARISNUM(2)) {
        var1n = VAR32(0);
        var2n = VAR32(2);
    } else {
        var1 = VAR(0);
        var2 = VAR(2);
        if(myisdechex_string(var1) && myisdechex_string(var2)) {
            var1 = NULL;
            var2 = NULL;
            var1n = VAR32(0);
            var2n = VAR32(2);
        }
        // in the For command I use a Set instruction at the beginning of the cycle with a String type
        // now the downside is that it's a bit slower but being used only at the beginning of the cycle there is no
        // loss of time (some milliseconds on tons of For cycles) and there is the pro of using also things like:
        //  for i = "hello" != "ciao"
    }

    res = -1;
    if(!strcmp(cond, "<")) {
        if(var1 && var2) {
            if(stricmp(var1, var2) < 0) res = 0;
        } else {
            if(var1n < var2n) res = 0;
        }
    } else if(!strcmp(cond, ">")) {
        if(var1 && var2) {
            if(stricmp(var1, var2) > 0) res = 0;
        } else {
            if(var1n > var2n) res = 0;
        }
    } else if(!strcmp(cond, "<>") || !strcmp(cond, "!=")) {
        if(var1 && var2) {
            if(stricmp(var1, var2) != 0) res = 0;
        } else {
            if(var1n != var2n) res = 0;
        }
    } else if(!strcmp(cond, "=") || !strcmp(cond, "==")) {
        if(var1 && var2) {
            if(stricmp(var1, var2) == 0) res = 0;
        } else {
            if(var1n == var2n) res = 0;
        }
    } else if(!strcmp(cond, ">=")) {
        if(var1 && var2) {
            if(stricmp(var1, var2) >= 0) res = 0;
        } else {
            if(var1n >= var2n) res = 0;
        }
    } else if(!strcmp(cond, "<=")) {
        if(var1 && var2) {
            if(stricmp(var1, var2) <= 0) res = 0;
        } else {
            if(var1n <= var2n) res = 0;
        }
    // added by me
    } else if(!strcmp(cond, "&")) {
        if(var1 && var2) {
            if(stristr(var1, var2)) res = 0;
        } else {
            if(var1n & var2n) res = 0;
        }
    } else if(!strcmp(cond, "^")) {
        if(var1 && var2) {
            if(!stricmp(var1, var2)) res = 0;
        } else {
            if(var1n ^ var2n) res = 0;
        }
    } else if(!strcmp(cond, "|")) {
        if(var1 && var2) {
            res = 0;
        } else {
            if(var1n | var2n) res = 0;
        }
    } else if(!strcmp(cond, "%")) {
        if(var1 && var2) {
            res = 0;
        } else {
            if(!var2n || (var1n % var2n)) res = 0;
        }
    } else if(!strcmp(cond, "/")) {
        if(var1 && var2) {
            res = 0;
        } else {
            if(!var2n || (var1n / var2n)) res = 0;
        }
    } else if(!strcmp(cond, "<<")) {
        if(var1 && var2) {
            res = 0;
        } else {
            if((u_int)var1n << (u_int)var2n) res = 0;
        }
    } else if(!strcmp(cond, ">>")) {
        if(var1 && var2) {
            res = 0;
        } else {
            if((u_int)var1n >> (u_int)var2n) res = 0;
        }
    } else if(!strcmp(cond, "!")) {
        if(var1 && var2) {
            res = 0;
        } else {
            if(!var1n) res = 0;
        }
    } else if(!strcmp(cond, "~")) {
        if(var1 && var2) {
            res = 0;
        } else {
            if(~var1n) res = 0;
        }
    } else if(!strcmp(cond, "ext") || !strcmp(cond, "extension")) {
        if(var1 && var2) {
            p = strrchr(var1, '.');
            if(p && !stricmp(p + 1, var2)) res = 0;
        } else {
            res = 0;
        }
    } else if(!strcmp(cond, "basename")) {
        if(var1 && var2) {
            p = strrchr(var1, '.');
            if(p) {
                *p = 0;
                if(!stricmp(var1, var2)) res = 0;
                *p = '.';
            }
        } else {
            res = 0;
        }
    } else {
        printf("\nError: invalid condition %s\n", cond);
        myexit(-1);
    }
    if(verbose) printf("             condition %s is%smet\n", cond, res ? " not " : " ");
    return(res);
}



int CMD_CLog_func(int cmd) {
    int     fd,
            offset,
            size,
            zsize;
    u8      *name;

    name    = VAR(0);
    offset  = VAR32(1);
    zsize   = VAR32(2);
    size    = VAR32(5);
    fd      = FILEZ(7);

    if(dumpa(fd, name, offset, size, zsize) < 0) return(-1);
    return(0);
}



int CMD_FindLoc_func(int cmd) {
    static u8   *sign   = NULL,
                *buff   = NULL;
    int     fd,
            i,
            len,
            oldoff,
            offset  = -1,
            str_len,
            sign_len;
    u8      *str,
            *s,
            *ret_if_error;

    fd      = FILEZ(3);
    oldoff  = myftell(fd);
    str     = STR(2);   // remember that str can be also a sequence of bytes, included 0x00!
    str_len = NUM(2);
    ret_if_error = STR(4);

    if(NUM(1) == TYPE_STRING) {
        sign_len = str_len;
        if(sign_len == -1) ALLOC_ERR;
        sign = realloc(sign, sign_len + 1);
        if(!sign) STD_ERR;
        memcpy(sign, str, sign_len);
        sign[sign_len] = 0;
    } else if(NUM(1) == TYPE_UNICODE) {
        sign_len = (str_len + 1) * 2;  // yeah includes also the NULL delimiter, boring unicode
        if(sign_len == -1) ALLOC_ERR;
        sign = realloc(sign, sign_len + 1);
        if(!sign) STD_ERR;

        s = sign;
        for(i = 0; i < str_len; i++) {
            if(endian == MYLITTLE_ENDIAN) {
                *s++ = str[i];
                *s++ = 0;
            } else {
                *s++ = 0;
                *s++ = str[i];
            }
        }
        *s++ = 0;
        *s++ = 0;
    } else {
        sign_len = NUM(1);  // yeah the type in NUM(1) is written for having the size of the parameter, watch enum
        if(sign_len == -1) ALLOC_ERR;
        sign = realloc(sign, sign_len + 1);
        if(!sign) STD_ERR;
        putxx(sign, myatoi(str), sign_len);
    }
    if(sign_len <= 0) goto quit;

    if(!buff) {
        buff = malloc(BUFFSZ + 1);
        if(!buff) STD_ERR;
    }

    for(;;) {
        len = myfr(fd, buff, -1);   // -1 uses BUFFSZ automatically and doesn't quit if the file terminates
        if(len < sign_len) break;   // performes (len <= 0) too automatically
        for(i = 0; i <= (len - sign_len); i++) {
            if(!memcmp(buff + i, sign, sign_len)) {
                offset = ((int)myftell(fd) - len) + i;
                goto quit;
            }
        }
        myfseek(fd, sign_len - 1, SEEK_CUR);
    }

quit:
    myfseek(fd, oldoff, SEEK_SET);
    if(offset == -1) {
        if(ret_if_error) {
            add_var(CMD.var[0], NULL, ret_if_error, 0, -1);
        } else {
            return(-1); // confirmed
        }
    } else {
        add_var(CMD.var[0], NULL, NULL, offset, sizeof(int));
    }
    return(0);
}



int CMD_Get_func(int cmd) {
    int     tmpn    = 0,
            error   = 0;
    u8      *tmp    = NULL;

    tmp = myfrx(NUM(2), NUM(1), &tmpn, &error);
    // now tmp can be also NULL because myfrx is string/int
    //if(!tmp) return(-1);    // here should be good to quit... but I leave it as is for back compatibility with the old quickbms!
    if(error) return(-1);
    if(tmp) {
        add_var(CMD.var[0], NULL, tmp, 0, -1);
    } else {
        add_var(CMD.var[0], NULL, NULL, tmpn, sizeof(int));
    }
    return(0);
}



int CMD_IDString_func(int cmd) {
    static int  buffsz  = 0;
    static u8   *sign   = NULL,
                *buff   = NULL;
    int     fd,
            len;

    fd   = FILEZ(0);
    sign = STR(1);
    len  = NUM(1);
    if(len == -1) ALLOC_ERR;
    myalloc(&buff, len + 1, &buffsz);
    myfr(fd, buff, len);
    if(memcmp(buff, sign, len)) return(-1);
    return(0);
}



int CMD_GoTo_func(int cmd) {
    int     fd,
            pos;
    u8      *str;

    fd  = FILEZ(1);
    str = VAR(0);

    if(!stricmp(str, "SOF")) {
        myfseek(fd, 0, SEEK_SET);
    } else if(!stricmp(str, "EOF")) {
        myfseek(fd, 0, SEEK_END);
    } else {
        pos = VAR32(0);
        if(pos < 0) {
            myfseek(fd, pos, SEEK_END);
        } else {
            myfseek(fd, pos, SEEK_SET);
        }
    }
    return(0);
}



int CMD_SavePos_func(int cmd) {
    int     fd;

    fd  = FILEZ(1);
    add_var(CMD.var[0], NULL, NULL, myftell(fd), sizeof(int));
    return(0);
}



int rol(int n1, int n2) {
    return(((u_int)n1 << (u_int)n2) | ((u_int)n1 >> (u_int)(INTSZ - n2)));
}



int ror(int n1, int n2) {
    return(((u_int)n1 >> (u_int)n2) | ((u_int)n1 << (u_int)(INTSZ - n2)));
}



int bitswap(int n1, int n2) {
    int     out,
            rem = 0;

    if(n2 < INTSZ) {
        rem = n1 & (((int)-1) ^ (((int)1 << n2) - 1));
    }

    for(out = 0; n2; n2--) {
        out = (out << 1) | (n1 & 1);
        n1 = (u_int)n1 >> 1;
    }
    return(out | rem);
}



int byteswap(int n1, int n2) {
    int     out,
            rem = 0;

    if(n2 < (INTSZ >> 3)) {
        rem = n1 & (((int)-1) ^ (((int)1 << (n2 << 3)) - 1));
    }

    for(out = 0; n2; n2--) {
        out = (out << 8) | (n1 & 0xff);
        n1 = (u_int)n1 >> 8;
    }
    return(out | rem);
}



int math_operations(int var1, int op, int var2) {
    switch(op) {
        case '+': var1 += var2;                 break;
        case '*': var1 *= var2;                 break;
        case '/': if(!var2) { var1 = 0; } else { var1 /= var2; } break;
        case '-': var1 -= var2;                 break;
        case '^': var1 ^= var2;                 break;
        case '&': var1 &= var2;                 break;
        case '|': var1 |= var2;                 break;
        case '%': if(!var2) { var1 = 0; } else { var1 %= var2; } break;
        case '!': var1 = !var1;                 break;  // var1 only
        case '~': var1 = ~var1;                 break;  // var1 only
        case '<': var1 = var1 << var2;          break;  // do not use (u_int), check dh2004.bms
        case '>': var1 = var1 >> var2;          break;  // do not use (u_int), check dh2004.bms
        case 'l': var1 = rol(var1, var2);       break;
        case 'r': var1 = ror(var1, var2);       break;
        case 's': var1 = byteswap(var1, var2);  break;
        case 'w': var1 = bitswap(var1, var2);   break;
        case '=': var1 = var2;                  break;
        default: {
            printf("\nError: invalid operator \'%c\'\n", op);
            myexit(-1);
            break;
        }
    }
    return(var1);
}



int CMD_Math_func(int cmd) {
    int     op,
            var1,
            var2;

    var1 = VAR32(0);
    op   = NUM(1);
    var2 = VAR32(2);

    var1 = math_operations(var1, op, var2);

    add_var(CMD.var[0], NULL, NULL, var1, sizeof(int));
    return(0);
}



int CMD_Log_func(int cmd) {
    int     fd,
            offset,
            size;
    u8      *name;

    name    = VAR(0);
    offset  = VAR32(1);
    size    = VAR32(2);
    fd      = FILEZ(5);

    if(dumpa(fd, name, offset, size, 0) < 0) return(-1);
    return(0);
}



int CMD_Next_func(int cmd) {
    int     var;

    if(CMD.var[0] < 0) return(0);   // like } in the C for(;;)
    var = VAR32(0);
    add_var(CMD.var[0], NULL, NULL, var + 1, sizeof(int));
    return(0);
}



int CMD_GetDString_func(int cmd) {
    static int  buffsz  = 0;
    static u8   *buff   = NULL;
    int     fd,
            size;

    fd   = FILEZ(2);
    size = VAR32(1);
    if(size == -1) ALLOC_ERR;
    myalloc(&buff, size + 1, &buffsz);
    myfr(fd, buff, size);
    buff[size] = 0;
    add_var(CMD.var[0], NULL, buff, 0, size);
    return(0);
}



int CMD_ReverseLong_func(int cmd) {
    int     n;

    n = VAR32(0);
    n = (((n & 0xff000000) >> 24) |
         ((n & 0x00ff0000) >>  8) |
         ((n & 0x0000ff00) <<  8) |
         ((n & 0x000000ff) << 24));
    add_var(CMD.var[0], NULL, NULL, n, sizeof(int));
    return(0);
}



int CMD_Set_func(int cmd) {
    static int  tmpsz   = 0;
    static u8   *tmp    = NULL;
    int     i,
            c,
            varn        = 0,
            varsz       = 0;
    u8      *var        = NULL,
            *p;

    if(NUM(1) == TYPE_UNICODE) {    // this is a particular exception for unicode which is enough boring to handle in some cases
        p = VAR(2);
        for(i = 0;; i++) {
            if(endian == MYLITTLE_ENDIAN) {
                c = p[0];
            } else {
                c = p[1];
            }
            if(!c) break;
            if(i >= tmpsz) {
                tmpsz += STRINGSZ;
                tmp = realloc(tmp, tmpsz + 1);
                if(!tmp) STD_ERR;
            }
            tmp[i] = c;
            p += 2;
        }
        if(!tmp) tmp = malloc(1);   // useful
        tmp[i] = 0;
        var   = tmp;
        varsz = -1;
    } else if(NUM(1) == TYPE_BINARY) {
        var   = STR(2);
        varsz = NUM(2);
    } else if(ISNUMTYPE(NUM(1))) { // number type
        varn  = VAR32(2);
        varsz = VARSZ(2);
    } else {
        var   = VAR(2);
        varsz = VARSZ(2);
    }

    if(CMD.var[0] < 0) {    // MEMORY_FILE
        dumpa_memory_file(&memory_file[-CMD.var[0]], var, varsz);
    } else {
        if(var) {
            add_var(CMD.var[0], NULL, var, 0, varsz);
        } else {
            add_var(CMD.var[0], NULL, NULL, varn, sizeof(int));
        }
    }
    return(0);
}



u8 *strristr(u8 *s1, u8 *s2) {
    int     s1n,
            s2n;
    u8      *p;

    if(!s1 || !s2) return(NULL);
    s1n = strlen(s1);
    s2n = strlen(s2);
    if(s2n > s1n) return(NULL);
    for(p = s1 + (s1n - s2n); p >= s1; p--) {
        if(!strnicmp(p, s2, s2n)) return(p);
    }
    return(NULL);
}



int CMD_String_func(int cmd) {
    static u8   *var1   = NULL;
    int     i,
            op,
            len1,
            len2,
            num     = 0;
    u8      *var2,
            *p;

    var1 = re_strdup(var1, VAR(0), NULL);   // needed for editing
    op   = NUM(1);
    var2 = VAR(2);

    len1 = strlen(var1);
    len2 = strlen(var2);
    if(myisdechex_string(var2)) num = myatoi(var2);
    if(len2) {
        switch(op) {
            case '+': {
                var1 = realloc(var1, len1 + len2 + 1);
                if(!var1) STD_ERR;
                strcpy(var1 + len1, var2);
                break;
            }
            case '-': { // I know that this is not the same method used in BMS but you can't "substract" a string from the end of another... it means nothing!
                if(num > 0) {
                    if(num <= len1) var1[len1 - num] = 0;
                } else if(num < 0) {
                    num = -num;
                    if(num <= len1) var1[num] = 0;
                } else {
                    while((p = (u8 *)stristr(var1, var2))) {
                        memmove(p, p + len2, strlen(p + len2) + 1);
                    }
                }
                break;
            }
            case '^': {
                for(i = 0; i < len1; i++) {
                    var1[i] ^= var2[i % len2];
                }
                break;
            }
            case '<': { // var1="thisisastring" var2="4" = "isastring"
                if(num > 0) {
                    if(num <= len1) {
                        p = var1 + num;
                        memmove(var1, p, strlen(p) + 1);
                    }
                } else {
                    printf("\nError: no string variable2 supported in String for operator %c\n", op);
                    myexit(-1);
                }
                break;
            }
            //case '/': 
            //case '*': 
            case '%': {
                if(num > 0) {
                    var1[len1 % num] = 0;
                } else {
                    printf("\nError: no string variable2 supported in String for operator %c\n", op);
                    myexit(-1);
                }
                break;
            }
            case '&': { // var1="thisisastring" var2="isa" = "isastring"
                //if(num > 0) {
                    //var1[len1 & num] = 0; // use % for this job, & means nothing
                //} else {
                    p = (u8 *)stristr(var1, var2);
                    if(p) memmove(var1, p, strlen(p) + 1);
                //}
                break;
            }
            case '|': { // var1="thisisastring" var2="isa" = "string"
                p = (u8 *)stristr(var1, var2);
                if(p) memmove(var1, p + len2, strlen(p + len2) + 1);
                break;
            }
            case '>': { // var1="thisisastring" var2="isa" = "this" (from end)
                if(num > 0) {
                    if(num <= len1) {
                        var1[len1 - num] = 0;
                    }
                } else {
                    p = (u8 *)strristr(var1, var2);
                    if(p) *p = 0;
                }
                break;
            }
            default: {
                printf("\nError: invalid operator %c\n", op);
                myexit(-1);
                break;
            }
        }
        add_var(CMD.var[0], NULL, var1, 0, -1);
    }
    return(0);
}



int CMD_ImpType_func(int cmd) {
    if(verbose) printf("- ImpType command %d ignored (not supported)\n", cmd);
    return(0);
}



int CMD_Open_func(int cmd) {
    static u8   *fname  = NULL;
    filenumber_t    *filez;
    int     fdnum;
    u8      current_dir[PATHSZ + 1],
            *fdir,
            *p;

    fdir    = VAR(0);
    fname   = re_strdup(fname, VAR(1), NULL);   // needed for modifying it
    fdnum   = NUM(2);

    getcwd(current_dir, PATHSZ);

    filez = &filenumber[0];     // everything is ever referred to the main file

    if(!stricmp(fdir, "FDDE")) {    // fname is the new extension
        fdir = input_folder;
        p = strrchr(filez->fullname, '.');
        if(p) {
            p++;
        } else {
            p = filez->fullname + strlen(filez->fullname);
        }
        fname = realloc(fname, (p - filez->fullname) + strlen(fname) + 1);
        if(!fname) STD_ERR;
        memmove(fname + (p - filez->fullname), fname, strlen(fname) + 1);
        memcpy(fname, filez->fullname, p - filez->fullname);
    } else if(!stricmp(fdir, "FDSE")) { // I don't know if this is correct because it's not documented!
        fdir = input_folder;
        p = strrchr(filez->fullname, '\\');
        if(!p) p = strrchr(filez->fullname, '/');
        if(p) {
            p++;
        } else {
            p = filez->fullname;
        }
        fname = realloc(fname, (p - filez->fullname) + strlen(fname) + 1);
        if(!fname) STD_ERR;
        memmove(fname + (p - filez->fullname), fname, strlen(fname) + 1);
        memcpy(fname, filez->fullname, p - filez->fullname);
    } else {
        // nothing to do
    }

    if(fname[0] == '?') {
        printf("\n- you must choose the name of the other file to load:\n  ");
        fname = realloc(fname, PATHSZ + 1);
        if(!fname) STD_ERR;
        fgets(fname, PATHSZ, stdin);
        for(p = fname; *p && (*p != '\n') && (*p != '\r'); p++);
        *p = 0;
    }

    if(fdir[0]) {
        printf("- enter in folder %s\n", fdir);
        if(chdir(fdir) < 0) STD_ERR;
    }

    myfopen(fname, fdnum);

    chdir(current_dir); // return to the output folder
    return(0);
}



int CMD_GetCT_func(int cmd) {
    int     fd;
    u8      *tmp;

    fd = FILEZ(3);
    //if(NUM(1) < 0) {
        // ok
    //} else {
        //printf("\nError: GetCT is supported only with String type\n");
        //myexit(-1);
    //}
    tmp = fgetss(fd, VAR32(2), (NUM(1) == TYPE_UNICODE) ? 1 : 0, 0);
    if(!tmp) return(-1);    // compability with old quickbms
    add_var(CMD.var[0], NULL, tmp, 0, -1); // fgetss is handled as a string function at the moment
    return(0);
}



int CMD_ComType_func(int cmd) {
    u8      *str;

    str = STR(0);
    if(!stricmp(str, "zlib") || !stricmp(str, "zlib1")) {
        compression_type = COMP_ZLIB;
    } else if(!stricmp(str, "deflate") || !stricmp(str, "inflate")) {
        compression_type = COMP_DEFLATE;
    } else if(!stricmp(str, "lzo1")) {
        compression_type = COMP_LZO1;
    } else if(!stricmp(str, "lzo1a")) {
        compression_type = COMP_LZO1A;
    } else if(!stricmp(str, "lzo1b")) {
        compression_type = COMP_LZO1B;
    } else if(!stricmp(str, "lzo1c")) {
        compression_type = COMP_LZO1C;
    } else if(!stricmp(str, "lzo1f")) {
        compression_type = COMP_LZO1F;
    } else if(!stricmp(str, "lzo1x") || !stricmp(str, "lzo") || !stricmp(str, "minilzo") || !stricmp(str, "lzox")) {
        compression_type = COMP_LZO1X;
    } else if(!stricmp(str, "lzo1y")) {
        compression_type = COMP_LZO1Y;
    } else if(!stricmp(str, "lzo1z")) {
        compression_type = COMP_LZO1Z;
    } else if(!stricmp(str, "lzo2a")) {
        compression_type = COMP_LZO2A;
    } else if(!stricmp(str, "lzss")) {
        compression_type = COMP_LZSS;
    } else if(!stricmp(str, "lzssboh")) {
        compression_type = COMP_LZSSBOH;
    } else if(!stricmp(str, "lzx")) {
        compression_type = COMP_LZX;
    } else if(!stricmp(str, "gzip")) {
        compression_type = COMP_GZIP;
    } else if(!stricmp(str, "pkware") || !stricmp(str, "blast") || !stricmp(str, "explode") || !stricmp(str, "implode")) {
        compression_type = COMP_EXPLODE;
    } else if(!stricmp(str, "lzma")) {          // 5 bytes + lzma
        compression_type = COMP_LZMA;
    } else if(!stricmp(str, "lzma86head")) {    // 5 bytes + 8 bytes (size) + lzma
        compression_type = COMP_LZMA_86HEAD;
    } else if(!stricmp(str, "lzma86dec")) {     // 1 byte + 5 bytes + lzma
        compression_type = COMP_LZMA_86DEC;
    } else if(!stricmp(str, "lzma86dechead")) { // 1 byte + 5 bytes + 8 bytes (size) + lzma
        compression_type = COMP_LZMA_86DECHEAD;
    } else if(!stricmp(str, "bz2") || !stricmp(str, "bunzip") || !stricmp(str, "bzip")) {
        compression_type = COMP_BZ2;
    } else if(!stricmp(str, "XMemDecompress") || !stricmp(str, "XMEMCODEC_DEFAULT") || !stricmp(str, "XMEMCODEC_LZX") || !stricmp(str, "xcompress")) {
        compression_type = COMP_XMEMLZX;
    } else if(!stricmp(str, "hex") || !stricmp(str, "hexadecimal") || !stricmp(str, "hex2byte")) {
        compression_type = COMP_HEX;
    } else if(!stricmp(str, "base64")) {
        compression_type = COMP_BASE64;
    } else if(!stricmp(str, "unlzw") || !stricmp(str, "COM_LZW_Decompress")) {
        compression_type = COMP_LZW;
    } else if(!stricmp(str, "unlzwx") || !stricmp(str, "milestone_lzw")) {
        compression_type = COMP_LZWX;
    //} else if(!stricmp(str, "cab") || !stricmp(str, "mscab") || !stricmp(str, "mscf")) {
        //compression_type = COMP_CAB;
    //} else if(!stricmp(str, "chm") || !stricmp(str, "mschm") || !stricmp(str, "itsf")) {
        //compression_type = COMP_CHM;
    //} else if(!stricmp(str, "szdd")) {
        //compression_type = COMP_SZDD;
    } else if(!stricmp(str, "lzxcab") || !stricmp(str, "mslzx")) {
        compression_type = COMP_LZXCAB;
    } else if(!stricmp(str, "lzxchm")) {
        compression_type = COMP_LZXCHM;
    } else if(!stricmp(str, "rlew")) {
        compression_type = COMP_RLEW;
    } else if(!stricmp(str, "lzjb")) {
        compression_type = COMP_LZJB;
    } else if(!stricmp(str, "sfl_block")) {
        compression_type = COMP_SFL_BLOCK;
    } else if(!stricmp(str, "sfl_rle")) {
        compression_type = COMP_SFL_RLE;
    } else if(!stricmp(str, "sfl_nulls")) {
        compression_type = COMP_SFL_NULLS;
    } else if(!stricmp(str, "sfl_bits")) {
        compression_type = COMP_SFL_BITS;
    } else if(!stricmp(str, "lzma2")) {          // 1 bytes + lzma2
        compression_type = COMP_LZMA2;
    } else if(!stricmp(str, "lzma2_86head")) {    // 1 bytes + 8 bytes (size) + lzma2
        compression_type = COMP_LZMA2_86HEAD;
    } else if(!stricmp(str, "lzma2_86dec")) {     // 1 byte + 1 bytes + lzma2
        compression_type = COMP_LZMA2_86DEC;
    } else if(!stricmp(str, "lzma2_86dechead")) { // 1 byte + 1 bytes + 8 bytes (size) + lzma2
        compression_type = COMP_LZMA2_86DECHEAD;
    } else {
        printf("\nError: invalid compression type %s\n", str);
        myexit(-1);
    }
    return(0);
}



u8 *numbers_to_bytes(u8 *str, int *ret_size) {
    static int  buffsz  = 0;
    static u8   *buff   = NULL;
    int     i,
            len,
            num,
            size;

    if(ret_size) *ret_size = 0;
    for(i = 0; str[0]; i++) {
        while(*str && !myisalnum(*str)) str++;  // this one handles also dots, commas and other bad chars
        num = readbase(str, 10, &len);
        if(len <= 0) break;
        if(i >= buffsz) {
            buffsz += STRINGSZ;
            buff = realloc(buff, buffsz + 1);
            if(!buff) STD_ERR;
        }
        buff[i] = num;
        str += len;
    }
    if(buff) buff[i] = 0; // useless, only for possible new usages in future, return ret as NULL
    size = i;
    if(ret_size) *ret_size = size;
    if(verbose) {
        printf("- numbers_to_bytes of %d bytes\n ", size);
        for(i = 0; i < size; i++) printf(" 0x%02x", buff[i]);
        printf("\n");
    }
    return(buff);
}



int CMD_FileXOR_func(int cmd) {
    int     fd,
            pos_offset,
            curroff;
    u8      *tmp;

    if(CMD.var[0] >= 0) {
        NUMS2BYTES(VAR(0), CMD.num[1], CMD.str[0], CMD.num[0])
    }
    file_xor            = STR(0);
    file_xor_size       = NUM(1);
    if(!file_xor_size) {
        file_xor        = NULL;
        file_xor_pos    = NULL;
    } else {
        file_xor_pos    = &NUM(2);
        pos_offset      = VAR32(3);
        fd              = FILEZ(4); // not implemented
        if(pos_offset >= 0) {
            curroff = myftell(fd);
            if(curroff >= pos_offset) {
                (*file_xor_pos) = curroff - pos_offset;
            } else {
                (*file_xor_pos) = file_xor_size - ((pos_offset - curroff) % file_xor_size);
            }
        }
    }
    return(0);
}



int CMD_FileRot13_func(int cmd) {
    int     fd,
            pos_offset,
            curroff;
    u8      *tmp;

    if(CMD.var[0] >= 0) {
        NUMS2BYTES(VAR(0), CMD.num[1], CMD.str[0], CMD.num[0])
    }
    file_rot13          = STR(0);
    file_rot13_size     = NUM(1);
    if(!file_rot13_size) {
        file_rot13      = NULL;
        file_rot13_pos  = NULL;
    } else {
        file_rot13_pos  = &NUM(2);
        pos_offset      = VAR32(3);
        fd              = FILEZ(4); // not implemented
        if(pos_offset >= 0) {
            curroff = myftell(fd);
            if(curroff >= pos_offset) {
                (*file_rot13_pos) = curroff - pos_offset;
            } else {
                (*file_rot13_pos) = file_rot13_size - ((pos_offset - curroff) % file_rot13_size);
            }
        }
    }
    return(0);
}



int CMD_Strlen_func(int cmd) {
    add_var(CMD.var[0], NULL, NULL, strlen(VAR(1)), sizeof(int));
    return(0);
}



int CMD_GetVarChr_func(int cmd) {
    int     varsz,
            offset,
            fdnum,
            numsz,
            num;
    u8      *var;

    if(CMD.var[1] < 0) {
        fdnum = -CMD.var[1];
        if((fdnum <= 0) || (fdnum > MAX_FILES)) {
            printf("\nError: invalid MEMORY_FILE number in GetVarChr\n");
            myexit(-1);
        }
        var   = memory_file[fdnum].data;
        varsz = memory_file[fdnum].size;
    } else {
        var   = VAR(1);
        varsz = VARSZ(1);
    }
    offset = VAR32(2);
    numsz  = NUM(3);
    if(numsz < 0) {  // so anything but TYPE_LONG, TYPE_INT, TYPE_BYTE, TYPE_THREEBYTE
        printf("\nError: GetVarChr supports only the numerical types\n");
        myexit(-1);
    }

    if((offset < 0) || ((offset + numsz) > varsz)) {
        printf("\nError: offset in GetVarChr (0x%08x) is bigger than the var (0x%08x)\n", offset, varsz);
        myexit(-1);
    }

    num = getxx(var + offset, numsz);
    add_var(CMD.var[0], NULL, NULL, num, sizeof(int));
    return(0);
}



int CMD_PutVarChr_func(int cmd) {
    int     varsz,
            offset,
            fdnum,
            numsz,
            num;
    u8      *var;

    if(CMD.var[0] < 0) {
        fdnum = -CMD.var[0];
        if((fdnum <= 0) || (fdnum > MAX_FILES)) {
            printf("\nError: invalid MEMORY_FILE number in PutVarChr\n");
            myexit(-1);
        }
        var   = memory_file[fdnum].data;
        varsz = memory_file[fdnum].size;
    } else {
        var   = VAR(0);
        varsz = VARSZ(0);
    }
    offset = VAR32(1);
    num    = VAR32(2);
    numsz  = NUM(3);
    if(numsz < 0) {  // so anything but TYPE_LONG, TYPE_INT, TYPE_BYTE, TYPE_THREEBYTE
        printf("\nError: PutVarChr supports only the numerical types\n");
        myexit(-1);
    }

    if((offset < 0) || ((offset + numsz) > varsz)) {
        printf("\nError: offset in PutVarChr (0x%08x) is bigger than the var (0x%08x)\n", offset, varsz);
        myexit(-1);
    }

    putxx(var + offset, num, numsz);
    return(0);
}



int CMD_Padding_func(int cmd) {
    int     fd,
            tmp,
            size,
            offset;

    fd   = FILEZ(1);
    size = NUM(0);

    offset = myftell(fd);
    tmp = offset % size;
    if(tmp) myfseek(fd, size - tmp, SEEK_CUR);
    return(0);
}



int CMD_Encryption_func(int cmd) {
    int     keysz;
    u8      *type,
            *key,
            *ivec;

    // reset ANY ctx
    FREEX(aes_ctx,)
    FREEX(blowfish_ctx,)
    FREEX(des_ctx,)
    FREEX(des3_ctx,)
    FREEX(rc4_ctx,)
    FREEX(tea_ctx,)
    FREEX(xtea_ctx,)
    FREEX(idea_ctx,)
    FREEX(xor_ctx, free(xor_ctx->key))
    FREEX(rot_ctx, free(rot_ctx->key))
    FREEX(charset_ctx,)
    if(cmd < 0) return(0);  // bms init

    type  = VAR(0);
    key   = STR(1);
    keysz = NUM(1);
    ivec  = STR(2);

    encryption_ivec = ivec; // ivec can be NULL (ecb) or a string (CBC)

    if(!stricmp(type, "") || !keysz) {
        // do nothing, disables the encryption

    } else if(!stricmp(type, "aes") || !stricmp(type, "Rijndael")) {
        aes_ctx = malloc(sizeof(aes_context));
        if(!aes_ctx) STD_ERR;
        aes_setkey_dec(aes_ctx, key, keysz << 3);

    } else if(!stricmp(type, "blowfish")) {
        blowfish_ctx = malloc(sizeof(blf_ctx));
        if(!blowfish_ctx) STD_ERR;
        blf_key(blowfish_ctx, key, keysz);

    } else if(!stricmp(type, "des")) {
        des_ctx = malloc(sizeof(des_context));
        if(!des_ctx) STD_ERR;
        des_setkey_dec(des_ctx, key);

    } else if(!stricmp(type, "3des2") || !stricmp(type, "3des-112")) {
        des3_ctx = malloc(sizeof(des3_context));
        if(!des3_ctx) STD_ERR;
        des3_set2key_dec(des3_ctx, key);

    } else if(!stricmp(type, "3des") || !stricmp(type, "3des-168")) {
        des3_ctx = malloc(sizeof(des3_context));
        if(!des3_ctx) STD_ERR;
        des3_set3key_dec(des3_ctx, key);

    } else if(!stricmp(type, "rc4") || !stricmp(type, "arc4")) {
        rc4_ctx = malloc(sizeof(arc4_context));
        if(!rc4_ctx) STD_ERR;
        arc4_setup(rc4_ctx, key, keysz);

    } else if(!stricmp(type, "tea")) {
        tea_ctx = malloc(sizeof(tea_context));
        if(!tea_ctx) STD_ERR;
        tea_setup(tea_ctx, key);

    } else if(!stricmp(type, "xtea")) {
        xtea_ctx = malloc(sizeof(xtea_context));
        if(!xtea_ctx) STD_ERR;
        xtea_setup(xtea_ctx, key);

    } else if(!stricmp(type, "idea")) {
        idea_ctx = malloc(sizeof(IDEA_context));
        if(!idea_ctx) STD_ERR;
        do_setkey(idea_ctx, key, keysz);

    } else if(!stricmp(type, "xor")) {
        xor_ctx = malloc(sizeof(xor_context));
        if(!xor_ctx) STD_ERR;
        xor_setkey(xor_ctx, key, keysz);

    } else if(!stricmp(type, "rot") || !stricmp(type, "rot13")) {
        rot_ctx = malloc(sizeof(rot_context));
        if(!rot_ctx) STD_ERR;
        rot_setkey(rot_ctx, key, keysz);

    } else if(!stricmp(type, "charset") || !stricmp(type, "chartable") || !stricmp(type, "substitution")) {
        if(keysz != 256) {
            printf("\nError: the charset encryption key must be 256 bytes long, yours is %d\n", keysz);
            myexit(-1);
        }
        charset_ctx = malloc(sizeof(charset_context));
        if(!charset_ctx) STD_ERR;
        charset_setkey(charset_ctx, key, keysz);

    } else {
        printf("\nError: unsupported encryption type (%s)\n",  type);
        myexit(-1);
    }

    if(verbose && keysz) printf("- encryption with algorithm %s and key of %d bytes\n", type, keysz);
    return(0);
}



int CMD_Print_func(int cmd) {
    int     i,
            idx;
    u8      *p,
            *msg;

    msg = STR(0);
    printf("- SCRIPT's MESSAGE:\n");

    while(*msg) {
        printf("  ");
        for(i = 0; i < 77; i++) {
            if(!*msg) break;
            if(*msg == '%') {
                msg++;
                p = strchr(msg, '%');
                if(!p) continue;
                idx = get_var_from_name(msg, p - msg);
                if(idx < 0) continue;
                fputs(get_var(idx), stdout);
                msg = p + 1;
            } else {
                if(*msg == '\n') {
                    msg++;
                    break;
                }
                fputc(*msg, stdout);
                msg++;
            }
        }
        fputc('\n', stdout);
    }
    printf("\n");
    return(0);
}



int CMD_GetArray_func(int cmd) {
    int     index,
            array_num;

    //var       = VAR(0);
    array_num = NUM(1);
    index     = VAR32(2);

    if((array_num < 0) || (array_num >= MAX_ARRAYS)) {
        printf("\nError: this BMS script uses more arrays than how much supported\n");
        myexit(-1);
    }
    if((index < 0) || (index >= array[array_num].elements)) {
        printf("\nError: this BMS script uses more array elements than how much supported\n");
        myexit(-1);
    }

    add_var(CMD.var[0], NULL, array[array_num].str[index], 0, -1);
    return(0);
}



int CMD_PutArray_func(int cmd) {
    int     i,
            num,
            index,
            array_num;
    u8      *var;

    array_num = NUM(0);
    index     = VAR32(1);
    var       = VAR(2);

    if((array_num < 0) || (array_num >= MAX_ARRAYS)) {
        printf("\nError: this BMS script uses more arrays than how much supported\n");
        myexit(-1);
    }
    if((index < 0) /*|| (index >= array[array_num].elements)*/) {
        printf("\nError: this BMS script uses more array elements than how much supported\n");
        myexit(-1);
    }

    if(index >= array[array_num].elements) {
        num = (index + 1) * sizeof(u8 *);   // +1 is necessary
        if(num < index) ALLOC_ERR;
        array[array_num].str = realloc(array[array_num].str, num);
        if(!array[array_num].str) STD_ERR;
        for(i = array[array_num].elements; i <= index; i++) {   // <= remember!!! (example 0 and 0)
            array[array_num].str[i] = NULL;
        }
        array[array_num].elements = index + 1;
    }

    array[array_num].str[index] = re_strdup(array[array_num].str[index], var, NULL);
    return(0);
}



int CMD_Function_func(int start_cmd, int nop, int *ret_break) {
    variable_t  *newvar = NULL,
                *oldvar = NULL;
    int     ret,
            cmd,
            i;
    u8      *func_name;

    cmd = start_cmd;

    if(CMD.type != CMD_CallFunction) {   // quick skip
        for(cmd++; CMD.type != CMD_NONE; cmd++) {
            if(CMD.type == CMD_EndFunction) return(cmd);
            if(CMD.type == CMD_StartFunction) break;
        }
        printf("\nError: no EndFunction command found\n");
        myexit(-1);
    }
    if(nop) return(start_cmd);

    func_name = STR(0);
    for(cmd = 0;; cmd++) {
        if(CMD.type == CMD_NONE) {
            printf("\nError: the function %s has not been found\n", func_name);
            myexit(-1);
        }
        if((CMD.type == CMD_StartFunction) && !stricmp(func_name, STR(0))) break;
    }

    newvar = calloc(variables + 1, sizeof(variable_t));
    if(!newvar) STD_ERR;  // calloc is better so it zeroes also the last variable automatically
    for(i = 0; i < variables; i++) {    // duplicate the strings, the first NULL in re_strdup is NECESSARY!!!
        memcpy(&newvar[i], &variable[i], sizeof(variable_t));
        if(variable[i].name)  newvar[i].name  = re_strdup(NULL, variable[i].name,  NULL);   // not needed
        if(variable[i].value) newvar[i].value = re_strdup(NULL, variable[i].value, NULL);
    }
    oldvar   = variable;
    variable = newvar;

    ret = start_bms(cmd + 1, nop, ret_break);

    for(i = 0; i < variables; i++) {
        if(newvar[i].name)  free(newvar[i].name);
        if(newvar[i].value) free(newvar[i].value);
    }
    free(newvar);
    variable = oldvar;

    if(ret < 0) return(ret);
    return(start_cmd);
}



#undef strdup
files_t *add_files(u8 *fname, int fsize, int *ret_files) {
    static int      filesi  = 0,
                    filesn  = 0;
    static files_t  *files  = NULL;

    if(ret_files) {
        *ret_files = filesi;
        filesi = 0; // reset, do NOT free
        return(files);
    }

    if(filter_in_files && (check_wildcard(fname, filter_in_files) < 0)) return(NULL);

    if(filesi >= filesn) {
        filesn += 1024;
        files = realloc(files, sizeof(files_t) * filesn);
        if(!files) STD_ERR;
    }
    files[filesi].name   = strdup(fname);
    files[filesi].offset = 0;
    files[filesi].size   = fsize;
    filesi++;
    return(NULL);
}
#define strdup          strdup_error



int recursive_dir(u8 *filedir) {
    int     plen,
            ret     = -1;

#ifdef WIN32
    static int      winnt = -1;
    OSVERSIONINFO   osver;
    WIN32_FIND_DATA wfd;
    HANDLE  hFind;

    if(winnt < 0) {
        osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&osver);
        if(osver.dwPlatformId >= VER_PLATFORM_WIN32_NT) {
            winnt = 1;
        } else {
            winnt = 0;
        }
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "\\*.*");
    plen++;

    if(winnt) { // required to avoid problems with Vista and Windows7!
        hFind = FindFirstFileEx(filedir, FindExInfoStandard, &wfd, FindExSearchNameMatch, NULL, 0);
    } else {
        hFind = FindFirstFile(filedir, &wfd);
    }
    if(hFind == INVALID_HANDLE_VALUE) return(0);
    do {
        if(!strcmp(wfd.cFileName, ".") || !strcmp(wfd.cFileName, "..")) continue;

        strcpy(filedir + plen, wfd.cFileName);

        if(wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
            if(recursive_dir(filedir) < 0) goto quit;
        } else {
            add_files(filedir + 2, wfd.nFileSizeLow, NULL);
        }
    } while(FindNextFile(hFind, &wfd));
    ret = 0;

quit:
    FindClose(hFind);
#else
    struct  stat    xstat;
    struct  dirent  **namelist;
    int     n,
            i;

    n = scandir(filedir, &namelist, NULL, NULL);
    if(n < 0) {
        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            STD_ERR;
        }
        add_files(filedir + 2, xstat.st_size, NULL);
        return(0);
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "/");
    plen++;

    for(i = 0; i < n; i++) {
        if(!strcmp(namelist[i]->d_name, ".") || !strcmp(namelist[i]->d_name, "..")) continue;
        strcpy(filedir + plen, namelist[i]->d_name);

        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            STD_ERR;
        }
        if(S_ISDIR(xstat.st_mode)) {
            if(recursive_dir(filedir) < 0) goto quit;
        } else {
            add_files(filedir + 2, xstat.st_size, NULL);
        }
        free(namelist[i]);
    }
    ret = 0;

quit:
    for(; i < n; i++) free(namelist[i]);
    free(namelist);
#endif
    filedir[plen - 1] = 0;
    return(ret);
}



int CMD_ScanDir_func(int cmd) {
    static u8   filedir[PATHSZ + 1] = "";
    static int  total_files         = -1;
    static int  curr_file           = 0;
    static files_t *files           = NULL;
    int     i;
    u8      *path;

    path    = STR(0);
    if(!path) return(-1);
    if(!filedir[0]) {
        if(strcmp(path, ".")) {
            printf("\nError: at the moment the ScanDir function accepts only the \".\" as scan path\n");
            myexit(-1);
        }
        strncpy(filedir, path, sizeof(filedir));
        filedir[sizeof(filedir) - 1] = 0;
        recursive_dir(filedir);
        files = add_files(NULL, 0, &total_files);
        curr_file = 0;
    }
    if(curr_file < total_files) {
        add_var(CMD.var[1], NULL, files[curr_file].name, 0, -1);
        add_var(CMD.var[2], NULL, NULL, files[curr_file].size, sizeof(int));
        curr_file++;
    } else {
        add_var(CMD.var[1], NULL, "", 0, -1);
        add_var(CMD.var[2], NULL, NULL, -1, sizeof(int));
        if(files) {
            for(i = 0; i < total_files; i++) {
                free(files[i].name);
            }
            free(files);
            files = NULL;
        }
        filedir[0] = 0;
        total_files = -1;
    }
    return(0);
}



// the rule is simple: start_bms is executed for EACH recursive command like do, for, if
int start_bms(int startcmd, int nop, int *ret_break) {
#define NEW_START_BMS(B,X,Y) \
    cmd = B(X, Y, ret_break); \
    if(cmd < 0) goto quit_error; \
    if(*ret_break) { \
        *ret_break = 0; \
        nop = 1; \
    }

    int     cmd,
            tmp;
    u8      *error  = NULL;

    if(startcmd < 0) {
        cmd = 0;    // needed because it's the beginning
    } else {
        cmd = startcmd;
    }
    if(verbose) printf("             .start_bms start: %d %d %d\n", startcmd, nop, *ret_break);
    for(; CMD.type != CMD_NONE; cmd++) {
        //if(verbose && CMD.debug_line) printf("\n%08x %s%s\n", (int)myftell(filenumber[0]), CMD.debug_line, nop ? " (SKIP)" : "");
        if(verbose && CMD.debug_line && !nop) printf("\n%08x %s\n", (int)myftell(0), CMD.debug_line);

        switch(CMD.type) {
            case CMD_CLog: {
                if(nop) break;
                if(CMD_CLog_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_Do: {
                //if(nop) break;
                NEW_START_BMS(start_bms, cmd + 1, nop)
                break;
            }
            case CMD_FindLoc: {
                if(nop) break;
                if(CMD_FindLoc_func(cmd) < 0) {
                    error = "the searched string has not been found";
                    goto quit_error;
                }
                break;
            }
            case CMD_For: {
                //if(nop) break;
                NEW_START_BMS(start_bms, cmd + 1, nop)
                break;
            }
            case CMD_ForTo: {
                if(nop) break;
                if(check_condition(cmd) < 0) nop = 1;
                break;
            }
            case CMD_Get: {
                if(nop) break;
                if(CMD_Get_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_GetDString: {
                if(nop) break;
                if(CMD_GetDString_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_GoTo: {
                if(nop) break;
                if(CMD_GoTo_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_IDString: {
                if(nop) break;
                if(CMD_IDString_func(cmd) < 0) {
                    error = "the signature doesn't match";        
                    goto quit_error;
                }
                break;
            }
            case CMD_ImpType: {
                if(nop) break;
                if(CMD_ImpType_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_Log: {
                if(nop) break;
                if(CMD_Log_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_Math: {
                if(nop) break;
                if(CMD_Math_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_Next: {
                if(nop) goto quit;
                if(CMD_Next_func(cmd) < 0) goto quit_error;
                if(startcmd >= 0) cmd = startcmd - 1;   // due to "cmd++"
                break;
            }
            case CMD_Open: {
                if(nop) break;
                if(CMD_Open_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_SavePos: {
                if(nop) break;
                if(CMD_SavePos_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_Set: {
                if(nop) break;
                if(CMD_Set_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_While: {
                if(nop) goto quit;
                if(check_condition(cmd) < 0) goto quit;
                if(startcmd >= 0) cmd = startcmd - 1;     // due to "cmd++"
                break;
            }
            case CMD_String: {
                if(nop) break;
                if(CMD_String_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_CleanExit: {
                if(nop) break;  // don't touch
                error = "invoked the termination of the extraction (CleanExit)";
                goto quit_error;
                break;
            }
            case CMD_If: {
                //if(nop) break;
                tmp = 0;
                do {
                    if(!tmp && !check_condition(cmd)) {
                        NEW_START_BMS(start_bms, cmd + 1, nop)
                        tmp = 1;
                    } else {
                        NEW_START_BMS(start_bms, cmd + 1, 1)
                    }
                } while(CMD.type != CMD_EndIf);
                break;
            }
            case CMD_Elif: {
                if(nop) goto quit;
                goto quit;
                break;
            }
            case CMD_Else: {
                if(nop) goto quit;
                goto quit;
                break;
            }
            case CMD_EndIf: {
                if(nop) goto quit;
                goto quit;
                break;
            }
            case CMD_GetCT: {
                if(nop) break;
                if(CMD_GetCT_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_ComType: {
                if(nop) break;
                if(CMD_ComType_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_ReverseLong: {
                if(nop) break;
                if(CMD_ReverseLong_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_NOP: {
                if(nop) break;
                // no operation, do nothing
                break;
            }
            case CMD_Endian: {
                if(nop) break;
                endian = NUM(0);
                break;
            }
            case CMD_FileXOR: {
                if(nop) break;
                if(CMD_FileXOR_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_FileRot13: {
                if(nop) break;
                if(CMD_FileRot13_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_Break: {
                if(nop) break;  // like cleanexit, don't touch
                nop = 1;
                *ret_break = 1;
                break;
            }
            case CMD_Strlen: {
                if(nop) break;
                if(CMD_Strlen_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_GetVarChr: {
                if(nop) break;
                if(CMD_GetVarChr_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_PutVarChr: {
                if(nop) break;
                if(CMD_PutVarChr_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_Debug: {
                //if(nop) break;
                verbose = !verbose;
                break;
            }
            case CMD_Padding: {
                if(nop) break;
                if(CMD_Padding_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_Append: {
                if(nop) break;
                append_mode = !append_mode;
                break;
            }
            case CMD_Encryption: {
                if(nop) break;
                if(CMD_Encryption_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_Print: {
                if(nop) break;
                if(CMD_Print_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_GetArray: {
                if(nop) break;
                if(CMD_GetArray_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_PutArray: {
                if(nop) break;
                if(CMD_PutArray_func(cmd) < 0) goto quit_error;
                break;
            }
            case CMD_StartFunction: {
                //if(nop) break;
                NEW_START_BMS(CMD_Function_func, cmd, 1)
                break;
            }
            case CMD_CallFunction: {
                if(nop) break;
                NEW_START_BMS(CMD_Function_func, cmd, nop)
                break;
            }
            case CMD_EndFunction: {
                if(nop) goto quit;
                goto quit;
                break;
            }
            case CMD_ScanDir: {
                if(nop) break;
                if(CMD_ScanDir_func(cmd) < 0) goto quit_error;
                break;
            }
            default: {
                printf("\nError: invalid command %d\n", CMD.type);
                myexit(-1);
                break;
            }
        }
    }
    return(-1); // CMD_NONE
quit_error:
    if(verbose) printf("\nError: %s\n", error ? error : (u8 *)"something wrong during the extraction");
    //myexit(-1);
    return(-1);
quit:
    if(verbose) printf("             .start_bms end: %d %d %d (ret %d)\n", startcmd, nop, *ret_break, cmd);
    return(cmd);
}



#undef strdup
int parse_bms(FILE *fds) {
    int     cmd,
            argc;
    u8      *debug_line = NULL,
            *argument[MAX_ARGS + 1] = { NULL },
            *tmp;

    cmd = 0;
    for(;;) {       // do NOT use "continue;"!
        if(cmd >= MAX_CMDS) {
            printf("\nError: the BMS script uses more commands than how much supported by this tool\n");
            myexit(-1);
        }
        argc = bms_line(fds, argument, &debug_line);
        if(argc < 0) break; // means "end of file"
        if(!argc) continue; // means "no command", here is possible to use "continue"

        argc--; // remove command argument
        // remember that myatoi is used only for the file number, all the rest must be add_var

               if(!stricmp(ARG[0], "QuickBMSver")   && (argc >= 1)) {
            CMD.type   = CMD_NOP;
            if(calc_quickbms_version(ARG[1]) > quickbms_version) {
                printf("\n"
                    "Error: this script has been created for a newer version of QuickBMS (%s),\n"
                    "       you can download it from:\n"
                    "\n"
                    "         http://aluigi.org/papers.htm#quickbms\n"
                    "\n", ARG[1]);
                myexit(-1);
            }

        } else if(!stricmp(ARG[0], "CLog")          && (argc >= 4)) {
            CMD.type   = CMD_CLog;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // name
            CMD.var[1] = add_var(0, ARG[2], NULL, 0, -2);       // offset
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // compressed size
            CMD.var[3] = -1;                                    // offsetoffset
            CMD.var[4] = -1;                                    // resourcesizeoffset
            CMD.var[6] = -1;                                    // uncompressedsizeoffset
            if(argc >= 6) {
                CMD.var[5] = add_var(0, ARG[6], NULL, 0, -2);   // uncompressedsize
                CMD.num[7] = myatoifile(ARG[8]);                // filenumber
            } else {
                CMD.var[5] = add_var(0, ARG[4], NULL, 0, -2);   // uncompressedsize
                CMD.num[7] = myatoifile(ARG[5]);                // filenumber
            }

        } else if(
                 (!stricmp(ARG[0], "Do")            && (argc >= 0))
              || (!stricmp(ARG[0], "Loop")          && (argc >= 0))) {  // mex inifile (not BMS)
            CMD.type   = CMD_Do;

        } else if(!stricmp(ARG[0], "FindLoc")       && (argc >= 3)) {
            CMD.type   = CMD_FindLoc;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // var
            CMD.num[1] = add_datatype(ARG[2]);                  // datatype
            CSTRING(2, ARG[3])                                  // text/number
            CMD.num[3] = myatoifile(ARG[4]);                    // filenumber
            CMD.str[4] = strdup(ARG[5]);                        // optional/experimental: the value you want to return in case the string is not found

        } else if(!stricmp(ARG[0], "FindFileID")    && (argc >= 2)) {   // mex inifile (not BMS)
            CMD.type   = CMD_FindLoc;
            CMD.var[0] = add_var(0, ARG[2], NULL, 0, -2);       // var
            CMD.num[1] = add_datatype("String");                // datatype
            CMD.str[2] = strdup(ARG[1]);                        // text/number
            CMD.num[3] = myatoifile(ARG[3]);                    // filenumber

        } else if(!stricmp(ARG[0], "For")           && (argc >= 0)) {
            if(argc >= 3) {
                CMD.type   = CMD_Set;
                CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);   // VarName
                CMD.num[1] = add_datatype("String");            // datatype
                CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);   // Var/Number
                cmd++;
            }

            CMD.type   = CMD_For;   // yes, no arguments, this is the new way

            if(argc >= 5) {
                cmd++;
                CMD.type   = CMD_ForTo;
                CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);   // T
                                                                // = T_value (check later, it must be check_condition compatible)
                if(!stricmp(ARG[4], "To")) {                    // To
                    CMD.str[1] = strdup("<=");
                } else {
                    CMD.str[1] = strdup(ARG[4]);
                }
                CMD.var[2] = add_var(0, ARG[5], NULL, 0, -2);   // To_value
                //CMD.var[3] = add_var(0, ARG[3], NULL, 0, -2);  // T_value (not used)
            }

        } else if(!stricmp(ARG[0], "Get")           && (argc >= 2)) {
            CMD.type   = CMD_Get;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // varname
            CMD.num[1] = add_datatype(ARG[2]);                  // type
            CMD.num[2] = myatoifile(ARG[3]);                    // filenumber

        } else if(!stricmp(ARG[0], "GetLong")       && (argc >= 1)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Get;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // varname
            CMD.num[1] = add_datatype("Long");                  // type
            CMD.num[2] = myatoifile(ARG[2]);                    // filenumber

        } else if(!stricmp(ARG[0], "GetInt")        && (argc >= 1)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Get;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // varname
            CMD.num[1] = add_datatype("Int");                   // type
            CMD.num[2] = myatoifile(ARG[2]);                    // filenumber

        } else if(!stricmp(ARG[0], "GetByte")       && (argc >= 1)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Get;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // varname
            CMD.num[1] = add_datatype("Byte");                  // type
            CMD.num[2] = myatoifile(ARG[2]);                    // filenumber

        } else if(!stricmp(ARG[0], "GetString")     && (argc >= 2)) {   // mex inifile (not BMS)
            CMD.type   = CMD_GetDString;
            CMD.var[0] = add_var(0, ARG[2], NULL, 0, -2);       // varname
            CMD.var[1] = add_var(0, ARG[1], NULL, 0, -2);       // NumberOfCharacters
            CMD.num[2] = myatoifile(ARG[3]);                    // filenumber

        } else if(!stricmp(ARG[0], "GetNullString") && (argc >= 1)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Get;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // varname
            CMD.num[1] = add_datatype("String");                // type
            CMD.num[2] = myatoifile(ARG[2]);                    // filenumber

        } else if(!stricmp(ARG[0], "GetDString")    && (argc >= 2)) {
            CMD.type   = CMD_GetDString;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // varname
            CMD.var[1] = add_var(0, ARG[2], NULL, 0, -2);       // NumberOfCharacters
            CMD.num[2] = myatoifile(ARG[3]);                    // filenumber

        } else if(!stricmp(ARG[0], "GoTo")          && (argc >= 1)) {
            CMD.type   = CMD_GoTo;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // pos
            CMD.num[1] = myatoifile(ARG[2]);                    // file

        } else if(
                  (!stricmp(ARG[0], "IDString")     && (argc >= 1))
               || (!stricmp(ARG[0], "ID")           && (argc >= 1))) {  // mex inifile (not BMS)
            CMD.type   = CMD_IDString;
            if(argc == 1) {
                CMD.num[0] = 0;
                CSTRING(1, ARG[1])                              // string
            } else {
                CMD.num[0] = myatoifile(ARG[1]);                // filenumber
                if(CMD.num[0] == MAX_FILES) {                   // simple work-around to avoid the different syntax of idstring
                    CSTRING(1, ARG[1])                          // string
                    CMD.num[0] = myatoifile(ARG[2]);
                } else {
                    CSTRING(1, ARG[2])                          // string
                    // CMD.num[0] = myatoifile(ARG[1]); // already set
                }
            }

        } else if(!strnicmp(ARG[0], "ID=", 3)       && (argc >= 0)) {   // mex inifile (not BMS)
            CMD.type   = CMD_IDString;
            CMD.num[0] = 0;
            CMD.str[1] = strdup(ARG[0] + 3);                    // bytes

        } else if(!stricmp(ARG[0], "ImpType")       && (argc >= 1)) {
            CMD.type   = CMD_ImpType;
            CMD.str[0] = strdup(ARG[1]);                        // type

        } else if(!stricmp(ARG[0], "Log")           && (argc >= 3)) {
            CMD.type   = CMD_Log;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // name
            CMD.var[1] = add_var(0, ARG[2], NULL, 0, -2);       // offset
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // size
            CMD.var[3] = -1;                                    // offsetoffset
            CMD.var[4] = -1;                                    // resourcesizeoffset
            if(argc >= 5) {
                CMD.num[5] = myatoifile(ARG[6]);                // filenumber
            } else {
                CMD.num[5] = myatoifile(ARG[4]);                // filenumber
            }

        } else if(!stricmp(ARG[0], "ExtractFile")   && (argc >= 0)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Log;
            CMD.var[0] = add_var(0, "FILENAME", NULL, 0, -2);   //  name
            CMD.var[1] = add_var(0, "FILEOFF",  NULL, 0,  0);   // offset
            CMD.var[2] = add_var(0, "FILESIZE", NULL, 0, -2);   // size
            CMD.var[3] = -1;                                    // offsetoffset
            CMD.var[4] = -1;                                    // resourcesizeoffset
            CMD.num[5] = myatoifile(ARG[6]);                    // filenumber

        } else if(!stricmp(ARG[0], "Math")          && (argc >= 3)) {
            CMD.type   = CMD_Math;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // var1
            CMD.num[1] = ARG[2][0];                             // op
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // var2

        } else if(!stricmp(ARG[0], "Add")           && (argc >= 3)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Math;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // var1
            CMD.num[1] = '+';                                   // op (skip specifier!)
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // var2

        } else if(!stricmp(ARG[0], "Subst")         && (argc >= 3)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Math;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // var1
            CMD.num[1] = '-';                                   // op (skip specifier!)
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // var2

        } else if(!stricmp(ARG[0], "Multiply")      && (argc >= 5)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Set;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // VarName
            CMD.num[1] = add_datatype("String");                // datatype
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // Var/Number
            cmd++;
            CMD.type   = CMD_Math;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // var1
            CMD.num[1] = '*';                                   // op
            CMD.var[2] = add_var(0, ARG[5], NULL, 0, -2);       // var2

        } else if(!stricmp(ARG[0], "Divide")        && (argc >= 5)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Set;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // VarName
            CMD.num[1] = add_datatype("String");                // datatype
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // Var/Number
            cmd++;
            CMD.type   = CMD_Math;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // var1
            CMD.num[1] = '/';                                   // op
            CMD.var[2] = add_var(0, ARG[5], NULL, 0, -2);       // var2

        } else if(!stricmp(ARG[0], "Up")            && (argc >= 1)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Math;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // var1
            CMD.num[1] = '+';                                   // op (skip specifier!)
            CMD.var[2] = add_var(0, "1", NULL, 0, -2);          // var2

        } else if(!stricmp(ARG[0], "Down")          && (argc >= 1)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Math;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // var1
            CMD.num[1] = '-';                                   // op (skip specifier!)
            CMD.var[2] = add_var(0, "1", NULL, 0, -2);          // var2

        } else if(!stricmp(ARG[0], "Next")          && (argc >= 0)) {
            CMD.type   = CMD_Next;
            if(!argc) {
                CMD.var[0] = -1;
            } else {
                CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);   // VarName
            }

        } else if(!stricmp(ARG[0], "Open")          && (argc >= 2)) {
            CMD.type   = CMD_Open;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // Folder/Specifier
            CMD.var[1] = add_var(0, ARG[2], NULL, 0, -2);       // Filename/Extension
            CMD.num[2] = myatoifile(ARG[3]);                    // File (default is 0, the same file)

        } else if(!stricmp(ARG[0], "SavePos")       && (argc >= 1)) {
            CMD.type   = CMD_SavePos;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // VarName
            CMD.num[1] = myatoifile(ARG[2]);                    // File

        } else if(!stricmp(ARG[0], "Set")           && (argc >= 2)) {
            CMD.type   = CMD_Set;
            //CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);      // VarName
            if(!strnicmp(ARG[1], MEMORY_FNAME, MEMORY_FNAMESZ)) {
                CMD.var[0] = get_memory_file(ARG[1]);
            } else {
                CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);
            }
            if(argc == 2) {
                CMD.num[1] = add_datatype("String");            // datatype
                //CMD.var[2] = add_var(0, ARG[2], NULL, 0, -2);  // Var/Number
                tmp = ARG[2];
            } else {
                CMD.num[1] = add_datatype(ARG[2]);              // datatype
                //CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);  // Var/Number
                tmp = ARG[3];
            }
            if(CMD.num[1] == TYPE_BINARY) {
                CSTRING(2, tmp)
            } else {
                CMD.var[2] = add_var(0, tmp, NULL, 0, -2);
            }

        } else if(!stricmp(ARG[0], "SETFILECNT")    && (argc >= 1)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Set;
            CMD.var[0] = add_var(0, "FILECNT", NULL, 0, -2);    // VarName
            CMD.num[1] = add_datatype("String");                // datatype
            CMD.var[2] = add_var(0, ARG[1], NULL, 0, -2);       // Var/Number

        } else if(!stricmp(ARG[0], "SETBYTESREAD")  && (argc >= 1)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Set;
            CMD.var[0] = add_var(0, "BYTESREAD", NULL, 0, -2);  // VarName
            CMD.num[1] = add_datatype("String");                // datatype
            CMD.var[2] = add_var(0, ARG[1], NULL, 0, -2);       // Var/Number

        } else if(!stricmp(ARG[0], "While")         && (argc >= 3)) {
            CMD.type   = CMD_While;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // Varname
            CMD.str[1] = strdup(ARG[2]);                        // Criterium
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // VarName2

        } else if(!stricmp(ARG[0], "EndLoop")       && (argc >= 2)) {   // mex inifile (not BMS)
            CMD.type   = CMD_While;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // Varname
            CMD.str[1] = strdup("!=");                          // Criterium
            CMD.var[2] = add_var(0, ARG[2], NULL, 0, -2);       // VarName2

        } else if(!stricmp(ARG[0], "String")        && (argc >= 3)) {
            CMD.type   = CMD_String;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // VarName1
            CMD.num[1] = ARG[2][0];                             // op
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // VarName2

        } else if(!stricmp(ARG[0], "CleanExit")     && (argc >= 0)) {
            CMD.type   = CMD_CleanExit;

        } else if(!stricmp(ARG[0], "Case")          && (argc >= 2)) {   // mex inifile (not BMS)
            CMD.type   = CMD_If;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // VarName1
            CMD.str[1] = strdup("=");                           // Criterium
            CMD.var[2] = add_var(0, ARG[2], NULL, 0, -2);       // VarName2
            cmd++;
            CMD.type   = CMD_EndIf;

        } else if(!stricmp(ARG[0], "If")            && (argc >= 3)) {
            CMD.type   = CMD_If;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // VarName1
            CMD.str[1] = strdup(ARG[2]);                        // Criterium
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // VarName2

        } else if(!stricmp(ARG[0], "Elif")          && (argc >= 3)) {   // copy as above!
            CMD.type   = CMD_Elif;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // VarName1
            CMD.str[1] = strdup(ARG[2]);                        // Criterium
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // VarName2

        } else if(!stricmp(ARG[0], "Else")          && (argc >= 0)) {
            CMD.type   = CMD_Else;
            if((argc >= 4) && !stricmp(ARG[1], "If")) {         // copy as above!
                CMD.type   = CMD_Elif;
                CMD.var[0] = add_var(0, ARG[2], NULL, 0, -2);   // VarName1
                CMD.str[1] = strdup(ARG[3]);                    // Criterium
                CMD.var[2] = add_var(0, ARG[4], NULL, 0, -2);   // VarName2
            }

        } else if(!stricmp(ARG[0], "EndIf")         && (argc >= 0)) {
            CMD.type   = CMD_EndIf;

        } else if(!stricmp(ARG[0], "GetCT")         && (argc >= 3)) {
            CMD.type   = CMD_GetCT;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // variable
            CMD.num[1] = add_datatype(ARG[2]);                  // datatype
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // character
            CMD.num[3] = myatoifile(ARG[4]);                    // filenumber

        } else if(!stricmp(ARG[0], "ComType")       && (argc >= 1)) {
            CMD.type   = CMD_ComType;
            CMD.str[0] = strdup(ARG[1]);                        // ComType

        } else if(
                 (!stricmp(ARG[0], "ReverseLong")   && (argc >= 1))
              || (!stricmp(ARG[0], "FlipLong")      && (argc >= 1))) {  // mex inifile (not BMS)
            CMD.type   = CMD_ReverseLong;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // variable

        } else if(!stricmp(ARG[0], "PROMPTUSER")    && (argc >= 0)) {   // mex inifile (not BMS)
            // do nothing, this command is useless
            CMD.type   = CMD_NOP;

        } else if(!stricmp(ARG[0], "EVENTS")        && (argc >= 0)) {   // mex inifile (not BMS)
            // do nothing, this command is useless
            CMD.type   = CMD_NOP;

        } else if(!stricmp(ARG[0], "SEPPATH")       && (argc >= 0)) {   // mex inifile (not BMS)
            // do nothing, this command is useless
            CMD.type   = CMD_NOP;

        } else if(!stricmp(ARG[0], "NOFILENAMES")   && (argc >= 0)) {   // mex inifile (not BMS)
            CMD.type   = CMD_Set;
            CMD.var[0] = add_var(0, "FILENAME", NULL, 0, -2);   // VarName
            CMD.num[1] = add_datatype("String");                // datatype
            CMD.var[2] = add_var(0, "", NULL, 0, -2);           // Var/Number

        } else if(!stricmp(ARG[0], "WriteLong")     && (argc >= 0)) {   // mex inifile (not BMS)
            // do nothing, this command is useless
            CMD.type   = CMD_NOP;

        } else if(!stricmp(ARG[0], "StrCReplace")   && (argc >= 0)) {   // mex inifile (not BMS)
            // do nothing, this command is useless
            CMD.type   = CMD_NOP;

        } else if(!stricmp(ARG[0], "StrEResizeC")   && (argc >= 0)) {   // mex inifile (not BMS)
            // do nothing, this command is useless
            CMD.type   = CMD_NOP;

        } else if(!stricmp(ARG[0], "SeperateHeader")&& (argc >= 0)) {   // mex inifile (not BMS)
            // do nothing, this command is useless
            CMD.type   = CMD_NOP;

        } else if(!stricmp(ARG[0], "Endian")        && (argc >= 1)) {
            CMD.type   = CMD_Endian;
            if(!stricmp(ARG[1], "little") || !stricmp(ARG[1], "intel") || !stricmp(ARG[1], "1234")) {
                CMD.num[0] = MYLITTLE_ENDIAN;
            } else if(!stricmp(ARG[1], "big") || !stricmp(ARG[1], "network") || !stricmp(ARG[1], "4321")) {
                CMD.num[0] = MYBIG_ENDIAN;
            } else {
                printf("\nError: invalid endian value %s\n", ARG[1]);
            }

        } else if(!stricmp(ARG[0], "FileXOR")       && (argc >= 1)) {
            CMD.type   = CMD_FileXOR;
            CMD.num[0] = 0; // used to contain the size of str[0], improves the performances
            if(myisdigit(ARG[1][0])) {
                NUMS2BYTES(ARG[1], CMD.num[1], CMD.str[0], CMD.num[0])
            } else {
                CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);   // string
            }
            CMD.num[2] = 0;                                     // reset pos
            if(argc == 1) {
                CMD.var[3] = add_var(0, "-1", NULL, 0, -2);     // current offset
                CMD.num[4] = 0;
            } else {
                CMD.var[3] = add_var(0, ARG[2], NULL, 0, -2);   // first position offset (used only for Log and multiple bytes in rare occasions)
                CMD.num[4] = myatoifile(ARG[3]);                // filenumber (not implemented)
            }

        } else if(!strnicmp(ARG[0], "FileRot", 7)   && (argc >= 1)) {
            CMD.type   = CMD_FileRot13;
            CMD.num[0] = 0; // used to contain the size of str[0], improves the performances
            if(myisdigit(ARG[1][0])) {
                NUMS2BYTES(ARG[1], CMD.num[1], CMD.str[0], CMD.num[0])
            } else {
                CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);   // string
            }
            CMD.num[2] = 0;                                     // reset pos
            if(argc == 1) {
                CMD.var[3] = add_var(0, "-1", NULL, 0, -2);     // current offset
                CMD.num[4] = 0;
            } else {
                CMD.var[3] = add_var(0, ARG[2], NULL, 0, -2);   // first position offset (used only for Log and multiple bytes in rare occasions)
                CMD.num[4] = myatoifile(ARG[3]);                // filenumber (not implemented)
            }

        } else if(!stricmp(ARG[0], "Break")         && (argc >= 0)) {
            CMD.type   = CMD_Break;

        } else if(!stricmp(ARG[0], "Strlen")        && (argc >= 2)) {
            CMD.type   = CMD_Strlen;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // dest var
            CMD.var[1] = add_var(0, ARG[2], NULL, 0, -2);       // string

        } else if(!stricmp(ARG[0], "GetVarChr")     && (argc >= 3)) {
            CMD.type   = CMD_GetVarChr;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // dst byte
            if(!strnicmp(ARG[2], MEMORY_FNAME, MEMORY_FNAMESZ)) {
                CMD.var[1] = get_memory_file(ARG[2]);
            } else {
                CMD.var[1] = add_var(0, ARG[2], NULL, 0, -2);   // src var
            }
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // offset
            if(argc == 3) {
                CMD.num[3] = add_datatype("byte");
            } else {
                CMD.num[3] = add_datatype(ARG[4]);
            }

        } else if(!stricmp(ARG[0], "PutVarChr")     && (argc >= 3)) {
            CMD.type   = CMD_PutVarChr;
            if(!strnicmp(ARG[1], MEMORY_FNAME, MEMORY_FNAMESZ)) {
                CMD.var[0] = get_memory_file(ARG[1]);;
            } else {
                CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);   // dst var
            }
            CMD.var[1] = add_var(0, ARG[2], NULL, 0, -2);       // offset
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // src byte
            if(argc == 3) {
                CMD.num[3] = add_datatype("byte");
            } else {
                CMD.num[3] = add_datatype(ARG[4]);
            }

        } else if(!stricmp(ARG[0], "Debug")         && (argc >= 0)) {
            CMD.type   = CMD_Debug;

        } else if(!stricmp(ARG[0], "Padding")       && (argc >= 1)) {
            CMD.type   = CMD_Padding;
            CMD.num[0] = myatoi(ARG[1]);                        // padding size
            CMD.num[1] = myatoifile(ARG[2]);                    // filenumber

        } else if(!stricmp(ARG[0], "Append")        && (argc >= 0)) {
            CMD.type   = CMD_Append;

        } else if(!stricmp(ARG[0], "Encryption")    && (argc >= 2)) {
            CMD.type   = CMD_Encryption;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // type
            if(!stricmp(ARG[2], "?")) {
                printf("\n"
                    "Error: seems that the script you are using needs that you specify a fixed\n"
                    "       %s key at line %d for using it, so edit the script source code\n"
                    "       adding this needed value, examples:\n"
                    "         encryption %s \"mykey\"\n"
                    "         encryption %s \"\\x6d\\x79\\x6b\\x65\\x79\"\n"
                    "\n", ARG[1], bms_line_number, ARG[1], ARG[1]);
                myexit(-1);
            }
            CSTRING(1, ARG[2])                                  // key
            if(argc > 2) {
                CSTRING(2, ARG[3])                              // ivec
            }

        } else if(!stricmp(ARG[0], "Print")         && (argc >= 1)) {
            CMD.type   = CMD_Print;
            CMD.str[0] = strdup(ARG[1]);                        // message
            cstring(CMD.str[0], CMD.str[0], -1, NULL);

        } else if(!stricmp(ARG[0], "GetArray")      && (argc >= 3)) {
            CMD.type   = CMD_GetArray;
            CMD.var[0] = add_var(0, ARG[1], NULL, 0, -2);       // var
            CMD.num[1] = myatoifile(ARG[2]);                    // array number
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // number/string

        } else if(!stricmp(ARG[0], "PutArray")      && (argc >= 3)) {
            CMD.type   = CMD_PutArray;
            CMD.num[0] = myatoifile(ARG[1]);                    // array number
            CMD.var[1] = add_var(0, ARG[2], NULL, 0, -2);       // number/string
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // var

        } else if(!stricmp(ARG[0], "StartFunction") && (argc >= 1)) {
            CMD.type   = CMD_StartFunction;
            CMD.str[0] = strdup(ARG[1]);

        } else if(!stricmp(ARG[0], "CallFunction")  && (argc >= 1)) {
            CMD.type   = CMD_CallFunction;
            CMD.str[0] = strdup(ARG[1]);

        } else if(!stricmp(ARG[0], "EndFunction")   && (argc >= 0)) {
            CMD.type   = CMD_EndFunction;
            //CMD.str[0] = strdup(ARG[1]);

        } else if(!stricmp(ARG[0], "ScanDir")       && (argc >= 3)) {
            CMD.type   = CMD_ScanDir;
            CMD.str[0] = strdup(ARG[1]);                        // path to scan
            CMD.var[1] = add_var(0, ARG[2], NULL, 0, -2);       // filename
            CMD.var[2] = add_var(0, ARG[3], NULL, 0, -2);       // filesize

        } else if(!stricmp(ARG[0], "Game") || !stricmp(ARG[0], "Archive")
               || !strnicmp(ARG[0], "Game ", 5)
               || !strnicmp(ARG[0], "Game:", 5)
               || !strnicmp(ARG[0], "Archive", 7)
               || !strnicmp(ARG[0], "Archive:", 8)
               || strstr(ARG[0], "-------")
               || strstr(ARG[0], "=-=-=-=")
               || strstr(ARG[0], "<bms")
               || strstr(ARG[0], "<bms>")
               || strstr(ARG[0], "</bms>")) {
            CMD.type   = CMD_NOP;

        } else {
            printf("\nError: invalid command \"%s\" or arguments %d at line %d\n", ARG[0], argc, bms_line_number);
            myexit(-1);
        }

        if(CMD.type == CMD_NONE) {
            printf("\nError: there is an error in this tool because there is no command type\n");
            myexit(-1);
        }
        CMD.debug_line = debug_line;
        cmd++;
    }
    for(variables = 0; variable[variables].name; variables++);
    return(0);
}
#define strdup          strdup_error



int myisalnum(int chr) {
    if((chr >= '0') && (chr <= '9')) return(1);
    if((chr >= 'a') && (chr <= 'z')) return(1);
    if((chr >= 'A') && (chr <= 'Z')) return(1);
    if(chr == '-') return(1);   // negative number
    return(0);
}



int myisdigitstr(u8 *str) { // only a quick version
    int     i;

    for(i = 0; str[i]; i++) {
        if(!i) {
            if(!myisdigit(str[i])) return(0);
        } else {
            if(!strchr("0123456789abcdefABCDEFx$", str[i])) return(0);
        }
    }
    return(1);
}



int myisdigit(int chr) {
    if((chr >= '0') && (chr <= '9')) return(1); // this is enough because hex start ever with 0x
    if(chr == '-') return(1);   // negative number
    return(0);
}



int myatoifile(u8 *str) {   // for quick usage
    int     fdnum;

    if(str && !strnicmp(str, MEMORY_FNAME, MEMORY_FNAMESZ)) {
        fdnum = get_memory_file(str);
    } else if(str && !strnicmp(str, "ARRAY", 5)) {
        fdnum = myatoi(str + 5);
    } else {
        if(!str || !str[0]) return(0);  // default is file number 0
        if(!myisdechex_string(str)) return(MAX_FILES);  // the syntax of idstring sux!
        fdnum = myatoi(str);
    }
    //if((fdnum <= 0) || (fdnum > MAX_FILES)) {
    if((fdnum < -MAX_FILES) || (fdnum > MAX_FILES)) {
        printf("\nError: invalid FILE number (%d)\n", fdnum);
        myexit(-1);
    }
    return(fdnum);
}



u8 *myitoa(int num) {
    static const u8 table[] = "0123456789abcdef";
    static u8       dst[3 + NUMBERSZ + 1] = "";
    u8      tmp[NUMBERSZ + 1],  // needed because hex numbers are inverted, I have already done various tests and this is the fastest!
            *p,                 // even faster than using directly dst as output
            *t;

    if(!num) {  // quick way, 0 is used enough often... ok it's probably useless
        dst[0] = '0';
        dst[1] = 0;
        return(dst);
    }

    p = dst;
    if(num < 0) {
        num = -num;
        *p++ = '-';
    }

    //if((num >= 0) && (num <= 9)) {  // quick solution for numbers under 10, so uses only one char, (num >= 0) avoids problems with 0x80000000
        //*p++ = table[num];
        //*p   = 0;
        //return(dst);
    //}

    t = tmp + (NUMBERSZ - 1);   // the -1 is needed (old tests)
    *t = 0;
    t--;
    if(decimal_notation) {
        do {   // "num" MUST be handled at the end of the cycle! example: 0
            *t = table[num % 10];
            num = (u_int)num / 10;
            if(!num) break;
            t--;
        } while(t >= tmp);
    } else {
        *p++ = '0'; // hex notation is better for debugging
        *p++ = 'x';
        do {   // "num" MUST be handled at the end of the cycle! example: 0
            *t = table[num & 15];
            num = (u_int)num >> 4;    // damn sign
            if(!num) break;
            t--;
        } while(t >= tmp);
    }
    strcpy(p, t);

    //sprintf(dst, "%d", num);  // old "one-instruction-only" solution, mine is better
    return(dst);
}



void mex_default_init(int file_only) {
    if(!file_only) EXTRCNT_idx   = add_var(0, "EXTRCNT", NULL, 0, sizeof(int));   // used by MultiEx as fixed variable
    BytesRead_idx = add_var(0, "BytesRead", NULL, 0, sizeof(int));   // used by MultiEx as fixed variable
    NotEOF_idx    = add_var(0, "NotEOF",    NULL, 1, sizeof(int));   // used by MultiEx as fixed variable
}



void bms_init(int reinit) {
    int     i,
            j;

    memset(filenumber,  0, sizeof(filenumber));
    variable = variable_main;
    memset(variable,    0, sizeof(variable_main));
    memset(command,     0, sizeof(command));
    memset(memory_file, 0, sizeof(memory_file));
    memset(array,       0, sizeof(array));
    for(i = 0; i < MAX_CMDS; i++) {
        for(j = 0; j < MAX_ARGS; j++) {
            command[i].var[j] = -0x7fffff;  // helps a bit to identify errors in this tool, DO NOT MODIFY IT! NEVER! (it's used in places like check_condition)
            command[i].num[j] = -0x7fffff;  // helps a bit to identify errors in this tool
            // do NOT touch command[i].str[j]
        }
    }
    if(mex_default) {
        mex_default_init(0);
    }
    CMD_Encryption_func(-1);

        bms_line_number     = 0;
        extracted_files     = 0;
        endian              = MYLITTLE_ENDIAN;
        //force_overwrite     = 0;
        variables           = 0;
        compression_type    = COMP_ZLIB;
        file_xor_pos        = NULL;
        file_xor_size       = 0;
        file_rot13_pos      = NULL;
        file_rot13_size     = 0;
        append_mode         = 0;
        file_xor            = NULL;
        file_rot13          = NULL;
        encryption_ivec     = NULL;
        EXTRCNT_idx         = 0;
        BytesRead_idx       = 0;
        NotEOF_idx          = 0;

    if(reinit) return;

    getcwd(input_folder, PATHSZ);
    quickbms_version = calc_quickbms_version(VER);
}



void bms_finish(void) { // totally useless function
    int     i,
            j;

    for(i = 0; i < MAX_FILES; i++) {
        if(filenumber[i].fd) fclose(filenumber[i].fd);
        if(filenumber[i].fullname) free(filenumber[i].fullname);
        if(filenumber[i].basename) free(filenumber[i].basename);
    }
    variable = variable_main;
    for(i = 0; i < MAX_VARS; i++) {
        if(variable[i].name)  free(variable[i].name);
        if(variable[i].value) free(variable[i].value);
    }
    for(i = 0; i < MAX_CMDS; i++) {
        if(command[i].debug_line) free(command[i].debug_line);
        for(j = 0; j < MAX_ARGS; j++) {
            if(command[i].str[j]) free(command[i].str[j]);
        }
    }
    for(i = 0; i < MAX_FILES; i++) {
        if(memory_file[i].data) free(memory_file[i].data);
    }
    for(i = 0; i < MAX_ARRAYS; i++) {
        for(j = 0; j < array[i].elements; j++) free(array[i].str[j]);
    }
    dumpa(0, NULL, -1, -1, -1);
    unzip(0, NULL, 0, NULL, 0);
    bms_line(NULL, NULL, NULL);
}



int bms_line(FILE *fd, u8 **argument, u8 **debug_line) {
    static  int buffsz  = 0;
    static  u8  *buff   = NULL;
    static  u8  tmpchars[NUMBERSZ + 1][MAX_ARGS] = {""};
    int     i,
            c;
    u8      tmp[1 + 1],
            *line,
            *p;

    if(!fd && !argument) {
        if(buff) free(buff);
        buff   = NULL;
        buffsz = 0;
        return(-1);
    }

    do {
        bms_line_number++;
        for(i = 0; ; i++) {
            c = fgetc(fd);
            if(c < 0) {
                if(!i) {    // end of file
                    bms_line_number = 0;
                    return(-1);
                }
                break;
            }
            if(c == '\n') break;
            if(i >= buffsz) {
                buffsz += STRINGSZ;
                buff = realloc(buff, buffsz + 1);
                if(!buff) STD_ERR;
            }
            buff[i] = c;
        }
        if(!buff) buff = malloc(1);
        buff[i] = 0;

        for(p = buff; *p && (*p != '\n') && (*p != '\r'); p++);
        *p = 0;

        for(p--; (p >= buff) && strchr(" \t;", *p); p--);
        p[1] = 0;

        for(p = buff; *p && strchr(" \t}", *p); p++);   // '}' is for C maniacs like me
        line = p;
        if(!myisalnum(line[0])) line[0] = 0;  // so we avoids both invalid chars and comments like # ; // and so on
    } while(!line[0]);

    if(debug_line) {
        *debug_line = malloc(32 + strlen(line) + 1);
        sprintf(*debug_line, "%-3d %s", bms_line_number, line);
        if(verbose) printf("READLINE %s\n", *debug_line);
    }

    for(i = 0; i < MAX_ARGS; i++) { // reset all
        argument[i] = NULL;
    }

    for(i = 0;;) {
        if(i >= MAX_ARGS) {
            printf("\nError: the BMS script uses more arguments than how much supported by this tool\n");
            myexit(-1);
        }
        for(p = line; *p && ((*p == ' ') || (*p == '\t')); p++);
        if(!*p) break;
        line = p;

        if(*line == '#') break;
        if(*line == ';') break;
        if(*line == '\'') {     // C char like 'A' or '\x41'
            line++;
            cstring(line, tmp, 1, &c);
            for(p = line + c; *p; p++) {
                if((p[0] == '\\') && (p[1] == '\'')) {
                    p++;
                    continue;
                }
                if(*p == '\'') break;
            }
            sprintf(tmpchars[i], "0x%02x", tmp[0]);
            argument[i] = tmpchars[i];
        } else if(*line == '\"') {  // string
            line++;
            for(p = line; *p; p++) {
                if((p[0] == '\\') && (p[1] == '\"')) {
                    p++;
                    continue;
                }
                if(*p == '\"') break;
            }
            argument[i] = line;
        } else {
            for(p = line; *p; p++) {
                if((*p == ' ') || (*p == '\t')) break;
            }
            argument[i] = line;
        }
        //if(p == line) break;  // this must be ignored otherwise "" is not handled
        i++;

        if(!*p) break;
        *p = 0;
        line = p + 1;
    }
    argument[i] = NULL;
    return(i);
}



int cstring(u8 *input, u8 *output, int maxchars, int *inlen) {
    int     n,
            len;
    u8      *p,
            *o;

    p = input;
    o = output;
    while(*p) {
        if(maxchars >= 0) {
            if((o - output) >= maxchars) break;
        }
        if(*p == '\\') {
            p++;
            switch(*p) {
                case 0:  return(-1); break;
                //case '0':  n = '\0'; break;
                case 'a':  n = '\a'; break;
                case 'b':  n = '\b'; break;
                case 'e':  n = '\e'; break;
                case 'f':  n = '\f'; break;
                case 'n':  n = '\n'; break;
                case 'r':  n = '\r'; break;
                case 't':  n = '\t'; break;
                case 'v':  n = '\v'; break;
                case '\"': n = '\"'; break;
                case '\'': n = '\''; break;
                case '\\': n = '\\'; break;
                case '?':  n = '\?'; break;
                case '.':  n = '.';  break;
                case 'x': {
                    //n = readbase(p + 1, 16, &len);
                    //if(len <= 0) return(-1);
                    if(sscanf(p + 1, "%02x%n", &n, &len) != 1) return(-1);
                    if(len > 2) len = 2;
                    p += len;
                    } break;
                default: {
                    //n = readbase(p, 8, &len);
                    //if(len <= 0) return(-1);
                    if(sscanf(p, "%3o%n", &n, &len) != 1) return(-1);
                    if(len > 3) len = 3;
                    p += (len - 1); // work-around for the subsequent p++;
                    } break;
            }
            *o++ = n;
        } else {
            *o++ = *p;
        }
        p++;
    }
    *o = 0;
    len = o - output;
    if(inlen) *inlen = p - input;
    return(len);
}



int perform_encryption(u8 *data, int datalen) {
    int     i;

    if(aes_ctx) {
        if(!encryption_ivec) {
            datalen /= 16;
            for(i = 0; i < datalen; i++) {
                aes_crypt_ecb(aes_ctx, AES_DECRYPT, data, data);
                data += 16;
            }
        } else {
            aes_crypt_cbc(aes_ctx, AES_DECRYPT, datalen, encryption_ivec, data, data);
        }

    } else if(blowfish_ctx) {
        if(!encryption_ivec) {
            blf_dec(blowfish_ctx, (void *)data, datalen / 8);
        } else {
            printf("\nError: CBC blowfish not supported\n");
            myexit(-1);
        }

    } else if(des_ctx) {
        if(!encryption_ivec) {
            datalen /= 8;
            for(i = 0; i < datalen; i++) {
                des_crypt_ecb(des_ctx, data, data);
                data += 8;
            }
        } else {
            des_crypt_cbc(des_ctx, DES_DECRYPT, datalen, encryption_ivec, data, data);
        }

    } else if(des3_ctx) {
        if(!encryption_ivec) {
            datalen /= 8;
            for(i = 0; i < datalen; i++) {
                des3_crypt_ecb(des3_ctx, data, data);
                data += 8;
            }
        } else {
            des3_crypt_cbc(des3_ctx, DES_DECRYPT, datalen, encryption_ivec, data, data);
        }

    } else if(rc4_ctx) {
        arc4_crypt(rc4_ctx, data, datalen);

    } else if(tea_ctx) {
        if(!encryption_ivec) {
            datalen /= 8;
            for(i = 0; i < datalen; i++) {
                tea_crypt(tea_ctx, TEA_DECRYPT, data, data);
                data += 8;
            }
        } else {
            printf("\nError: CBC tea not supported\n");
            myexit(-1);
        }

    } else if(xtea_ctx) {
        if(!encryption_ivec) {
            datalen /= 8;
            for(i = 0; i < datalen; i++) {
                xtea_crypt_ecb(xtea_ctx, XTEA_DECRYPT, data, data);
                data += 8;
            }
        } else {
            printf("\nError: CBC xtea not supported\n");
            myexit(-1);
        }

    } else if(idea_ctx) {
        datalen /= 8;
        for(i = 0; i < datalen; i++) {
            decrypt_block(idea_ctx, data, data);
            data += 8;
        }

    } else if(xor_ctx) {
        xor_crypt(xor_ctx, data, datalen);

    } else if(rot_ctx) {
        rot_crypt(rot_ctx, data, datalen);

    } else if(charset_ctx) {
        charset_crypt(charset_ctx, data, datalen);
    }
    return(0);
}



void dumpa_memory_file(memory_file_t *memfile, u8 *data, int size) {
    if(size == -1) ALLOC_ERR;
    if(append_mode) {
        memfile->pos   = memfile->size;
        memfile->size += size;
    } else {
        memfile->pos   = 0;
        memfile->size  = size;
    }
    if((u_int)memfile->size > (u_int)memfile->maxsize) {
        memfile->maxsize = memfile->size;
        if(memfile->maxsize == -1) ALLOC_ERR;
        memfile->data = realloc(memfile->data, memfile->maxsize + 1);
        if(!memfile->data) STD_ERR;
    }
    if(memfile->data) {
        memcpy(memfile->data + memfile->pos, data, size);
        memfile->data[memfile->pos + size] = 0;  // not needed, it's for a possible future usage or something else
    }
}



int dumpa(int fdnum, u8 *fname, int offset, int size, int zsize) {
    static  int insz    = 0,    // ONLY as total allocated input size
                outsz   = 0;    // ONLY as total allocated output size
    static  u8  *in     = NULL,
                *out    = NULL;

    memory_file_t   *memfile    = NULL;
    FILE    *fdo;
    int     tmp,
            oldoff,
            filetmp     = 0;
    u8      tmpname[NUMBERSZ + 4 + 1],
            *p;

    if(!fname && (offset < 0) && (size < 0)) {
        if(in)  free(in);
        if(out) free(out);
        in    = NULL;
        out   = NULL;
        insz  = 0;
        outsz = 0;
        return(-1);
    }

    // the following is a set of filename cleaning instructions to avoid that files or data with special names are not saved
    if(fname) {
        if(fname[1] == ':') fname += 2;
        for(p = fname; *p && (*p != '\n') && (*p != '\r'); p++) {
            if(strchr("?%*:|\"<>", *p)) {    // invalid filename chars not supported by the most used file systems
                *p = '_';
            }
        }
        *p = 0;
        for(p--; (p >= fname) && ((*p == ' ') || (*p == '.')); p--) *p = 0;   // remove final spaces and dots
    }

    if(!fname || !fname[0]) {
        fname = tmpname;
        sprintf(fname, "%08x.dat", extracted_files);
    }

    // handling of the output filename
    if(!strnicmp(fname, MEMORY_FNAME, MEMORY_FNAMESZ)) {
        memfile = &memory_file[-get_memory_file(fname)];    // yes, remember that it must be negative of negative
        if(verbose) printf("- create a memory file from offset %08x of %u bytes\n", offset, size);
    } else if(!stricmp(fname, TEMPORARY_FILE)) {
        filetmp = 1;
        if(verbose) printf("- create a temporary file from offset %08x of %u bytes\n", offset, size);
    } else {
        if(filter_files && (check_wildcard(fname, filter_files) < 0)) goto quit;
        printf("  %08x %-10u %s\n", offset, size, fname);
        if(listfd) {
            fprintf(listfd, "  %08x %-10u %s\n", offset, size, fname);
            fflush(listfd);
        }
    }

    if(list_only && !memfile && !filetmp) {
        // do nothing
    } else if((fname[strlen(fname) - 1] == '\\') || (fname[strlen(fname) - 1] == '/')) {    // folder
        // do nothing
    } else {
        fname = create_dir(fname);
        if(check_overwrite(fname) < 0) goto quit;

        oldoff = myftell(fdnum);
        myfseek(fdnum, offset, SEEK_SET);

        myalloc(&out, size, &outsz);
        if((zsize > 0) && (size > 0)) { // remember that the (size == zsize) check is NOT valid so can't be used in a "generic" way!
            myalloc(&in, zsize, &insz);
            myfr(fdnum, in, zsize);
            perform_encryption(in, zsize);

            switch(compression_type) {
                case COMP_ZLIB: {
                    size = unzip(15, in, zsize, out, size);
                    break;
                }
                case COMP_DEFLATE: {
                    size = unzip(-15, in, zsize, out, size);
                    break;
                }
                case COMP_LZO1:
                case COMP_LZO1A:
                case COMP_LZO1B:
                case COMP_LZO1C:
                case COMP_LZO1F:
                case COMP_LZO1X:
                case COMP_LZO1Y:
                case COMP_LZO1Z:
                case COMP_LZO2A: {
                    size = unlzo(in, zsize, out, size, compression_type);
                    break;
                }
                case COMP_LZSS: {
                    size = unlzss(out, size, in, zsize);
                    break;
                }
                case COMP_LZSSBOH: {
                    size = unlzss(in, zsize, out, size);
                    break;
                }
                case COMP_LZX: {
                    size = unlzx(in, zsize, out, size);
                    break;
                }
                case COMP_GZIP: {
                    size = ungzip(in, zsize, &out, &outsz); // outsz and NOT size because must be reallocated
                    break;
                }
                case COMP_EXPLODE: {
                    size = unexplode(in, zsize, out, size);
                    break;
                }
                case COMP_LZMA: {
                    size = unlzma(in, zsize, &out, size, LZMA_FLAGS_NONE, NULL);
                    break;
                }
                case COMP_LZMA_86HEAD: {
                    size = unlzma(in, zsize, &out, size, LZMA_FLAGS_86_HEADER, &outsz); // contains the uncompressed size
                    break;
                }
                case COMP_LZMA_86DEC: {
                    size = unlzma(in, zsize, &out, size, LZMA_FLAGS_86_DECODER, NULL);
                    break;
                }
                case COMP_LZMA_86DECHEAD: {
                    size = unlzma(in, zsize, &out, size, LZMA_FLAGS_86_DECODER | LZMA_FLAGS_86_HEADER, &outsz); // contains the uncompressed size
                    break;
                }
                case COMP_BZ2: {
                    size = unbzip2(in, zsize, out, size);
                    break;
                }
                case COMP_XMEMLZX: {
                    size = unxmemlzx(in, zsize, out, size);
                    break;
                }
                case COMP_HEX: {
                    size = unhex(in, zsize, out, size);
                    break;
                }
                case COMP_BASE64: {
                    size = unbase64(in, zsize, out, size);
                    break;
                }
                case COMP_LZW: {
                    size = unlzw(out, size, in, zsize);
                    break;
                }
                case COMP_LZWX: {
                    size = unlzwx(out, size, in, zsize);
                    break;
                }
                //case COMP_CAB: {
                    //size = unmspack_cab(in, zsize, out, size);
                    //break;
                //}
                //case COMP_CHM: {
                    //size = unmspack_chm(in, zsize, out, size);
                    //break;
                //}
                //case COMP_SZDD: {
                    //size = unmspack_szdd(in, zsize, out, size);
                    //break;
                //}
                case COMP_LZXCAB: {
                    size = unmslzx(in, zsize, out, size, 21, 0);
                    break;
                }
                case COMP_LZXCHM: {
                    size = unmslzx(in, zsize, out, size, 16, 2);
                    break;
                }
                case COMP_RLEW: {
                    size = unrlew(in, zsize, out, size);
                    break;
                }
                case COMP_LZJB: {
                    size = lzjb_decompress(in, out, zsize, size);
                    break;
                }
                case COMP_SFL_BLOCK: {
                    tmp = expand_block(in, out, zsize);
                    if(tmp > size) exit(1); // avoid hof
                    size = tmp;
                    break;
                }
                case COMP_SFL_RLE: {
                    tmp = expand_rle(in, out, zsize);
                    if(tmp > size) exit(1); // avoid hof
                    size = tmp;
                    break;
                }
                case COMP_SFL_NULLS: {
                    tmp = expand_nulls(in, out, zsize);
                    if(tmp > size) exit(1); // avoid hof
                    size = tmp;
                    break;
                }
                case COMP_SFL_BITS: {
                    tmp = expand_bits(in, out, zsize);
                    if(tmp > size) exit(1); // avoid hof
                    size = tmp;
                    break;
                }
                case COMP_LZMA2: {
                    size = unlzma2(in, zsize, &out, size, LZMA_FLAGS_NONE, NULL);
                    break;
                }
                case COMP_LZMA2_86HEAD: {
                    size = unlzma2(in, zsize, &out, size, LZMA_FLAGS_86_HEADER, &outsz); // contains the uncompressed size
                    break;
                }
                case COMP_LZMA2_86DEC: {
                    size = unlzma2(in, zsize, &out, size, LZMA_FLAGS_86_DECODER, NULL);
                    break;
                }
                case COMP_LZMA2_86DECHEAD: {
                    size = unlzma2(in, zsize, &out, size, LZMA_FLAGS_86_DECODER | LZMA_FLAGS_86_HEADER, &outsz); // contains the uncompressed size
                    break;
                }
                default: {
                    printf("\nError: unsupported compression type %d\n", compression_type);
                    break;
                }
            }
            if(size < 0) {
                printf("\n"
                    "Error: there is an error with the decompression\n"
                    "       the returned output size is negative (%d)\n", size);
                myexit(-1);
            }
            // do NOT add checks which verify if the unpacked size is like the expected one, I prefer the compatibility
        } else {
            myfr(fdnum, out, size);
            perform_encryption(out, size);
        }

        if(memfile) {
            dumpa_memory_file(memfile, out, size);
        } else {
            if(append_mode) {
                fdo = fopen(fname, "ab");
                if(!fdo) STD_ERR;
            } else {
                fdo = fopen(fname, "wb");
                if(!fdo) STD_ERR;
            }
            if(fwrite(out, 1, size, fdo) != size) {
                printf("\nError: impossible to write the file on the disk, check your space\n");
                myexit(-1);
            }
            fclose(fdo);
        }

        myfseek(fdnum, oldoff, SEEK_SET);
    }
    if(!memfile) {
        extracted_files++;
        if(mex_default) {
            add_var(EXTRCNT_idx, NULL, NULL, extracted_files, sizeof(int));
        }
    }
quit:
    return(0);
}



int hex2byte(u8 *hex) {
    static const signed char hextable[256] =
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\xff\xff\xff\xff\xff\xff"
        "\xff\x0a\x0b\x0c\x0d\x0e\x0f\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\x0a\x0b\x0c\x0d\x0e\x0f\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff";

    if((hextable[hex[0]] < 0) || (hextable[hex[1]] < 0)) return(-1);
    return((hextable[hex[0]] << 4) | hextable[hex[1]]);
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    if(!wildcard) return(0);
    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



u8 *create_dir(u8 *fname) {
    u8      *p,
            *l;

    p = strchr(fname, ':'); // unused
    if(p) {
        *p = '_';
        fname = p + 1;
    }
    for(p = fname; *p && strchr("\\/. \t:", *p); p++) *p = '_';
    fname = p;

    for(p = fname; ; p = l + 1) {
        for(l = p; *l && (*l != '\\') && (*l != '/'); l++);
        if(!*l) break;
        *l = 0;

        if(!strcmp(p, "..")) {
            p[0] = '_';
            p[1] = '_';
        }

        make_dir(fname);
        *l = PATHSLASH;
    }
    return(fname);
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    if(force_overwrite) return(0);
    if(!fname) return(0);
    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("- the file \"%s\" already exists\n  do you want to overwrite it (y/N/all)? ", fname);
    if(append_mode) printf("\n"
        "  (remember that you are in append mode so be sure that the output folder was\n"
        "  empty otherwise the new data will be appended to the existent files!) ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) == 'y') return(0);
    if(tolower(ans[0]) == 'a') {
        force_overwrite = 1;
        return(0);
    }
    return(-1);
}



void myalloc(u8 **data, int wantsize, int *currsize) {
    if(wantsize < 0) {
        printf("\nError: the requested amount of bytes to allocate is negative (%d)\n", wantsize);
        myexit(-1);
    }
    if(!wantsize) return;
    if(wantsize <= *currsize) {
        if(*currsize > 0) return;
    }
    *data = realloc(*data, wantsize);
    if(!*data) STD_ERR;
    *currsize = wantsize;
}



int getxx(u8 *tmp, int bytes) {
    int     num;
    int     i;

    num = 0;
    for(i = 0; i < bytes; i++) {
        if(endian == MYLITTLE_ENDIAN) {
            num |= (tmp[i] << (i << 3));
        } else {
            num |= (tmp[i] << ((bytes - 1 - i) << 3));
        }
    }
    return(num);
}



int putxx(u8 *data, int num, int bytes) {
    int     i;

    for(i = 0; i < bytes; i++) {
        if(endian == MYLITTLE_ENDIAN) {
            data[i] = num >> (i << 3);
        } else {
            data[i] = num >> ((bytes - 1 - i) << 3);
        }
    }
    return(bytes);
}



u8 *fgetss(int fdnum, int chr, int unicode, int line) {  // reads a chr terminated string, at the moment unicode is referred to the 16bit unicode
    static int  buffsz  = 0;
    static u8   *buff   = NULL;
    int     i,
            c,
            unicnt  = -1;

    // if(!fd) do nothing, modify myfgetc
    for(i = 0;;) {
        c = myfgetc(fdnum);
        if(c < 0) {
            if(!i) return(NULL);    // return a NULL if EOF... this is for compatibility with old versions of quickbms although it's not so right
            break;
        }
        if(unicode) {
            unicnt++;
            if(endian == MYLITTLE_ENDIAN) {
                if(unicnt & 1) continue;
            } else {
                if(!(unicnt & 1)) continue;
            }
        }
        if(c == chr) break;
        if(i >= buffsz) {
            buffsz += STRINGSZ;
            buff = realloc(buff, buffsz + 1);
            if(!buff) STD_ERR;
        }
        buff[i] = c;
        i++;
    }
    if(unicode) {
        if(endian == MYLITTLE_ENDIAN) c = myfgetc(fdnum);  // needed for reaching the real end of the unicode string (16 bit)
    }
    //if(c < 0) return(NULL);
    if(!buff) buff = malloc(1); // remember, anything returned by this function MUST be allocated
    buff[i] = 0;
    if(line) {
        for(i = 0; buff[i]; i++) {  // buff has been nulled
            if((buff[i] == '\r') || (buff[i] == '\n')) buff[i] = 0;
        }
    }
    return(buff);
}



int myfgetc(int fdnum) {
    memory_file_t   *memfile    = NULL;
    int     c;
    u8      buff[1];

    if(fdnum < 0) {
        memfile = &memory_file[-fdnum];
        if((memfile->pos + 1) > memfile->size) {
            c = -1;
        } else {
            c = memfile->data[memfile->pos];
            memfile->pos++;
        }
    } else {
        if(!filenumber[fdnum].fd) {
            printf("\nError: the specified file number (%d) has not been opened yet\n", fdnum);
            myexit(-1);
        }
        c = fgetc(filenumber[fdnum].fd);
    }
    if(c < 0) return(c);
    buff[0] = c;
    post_fread_actions(fdnum, buff, 1);
    return(buff[0]);
}



int fgetxx(int fdnum, int bytes) {
    u8      tmp[bytes];

    // if(!fd) do nothing, modify myfr
    myfr(fdnum, tmp, bytes);
    return(getxx(tmp, bytes));
}



int myfilesize(int fdnum) {
    struct stat xstat;

    if(fdnum < 0) {
        return(memory_file[-fdnum].size);
    }
    if(!filenumber[fdnum].fd) {
        printf("\nError: the specified file number (%d) has not been opened yet\n", fdnum);
        myexit(-1);
    }
    fstat(fileno(filenumber[fdnum].fd), &xstat);
    return(xstat.st_size);
}



u8 *myfrx(int fdnum, int type, int *ret_num, int *error) {
    int     retn    = 0;
    u8      *ret    = NULL;

    *error = 0;
    switch(type) {
        case TYPE_LONG:         retn = fgetxx(fdnum, 4);            break;
        case TYPE_INT:          retn = fgetxx(fdnum, 2);            break;
        case TYPE_BYTE:         retn = fgetxx(fdnum, 1);            break;
        case TYPE_THREEBYTE:    retn = fgetxx(fdnum, 3);            break;
        case TYPE_ASIZE:        retn = myfilesize(fdnum);           break;
        case TYPE_STRING:
            ret  = fgetss(fdnum, 0,    0, 0);
            if(!ret) *error = 1;    // this damn error stuff is needed for compatibility with the old quickbms
            break;                  // and located here doesn't affect the performances
        case TYPE_LINE:
            ret  = fgetss(fdnum, '\n', 0, 1);
            if(!ret) *error = 1;
            break;
        case TYPE_FILENAME:
            ret  = filenumber[fdnum].filename;
            if(!ret) *error = 1;
            break;
        case TYPE_BASENAME:
            ret  = filenumber[fdnum].basename;
            if(!ret) *error = 1;
            break;
        case TYPE_EXTENSION:
            ret  = filenumber[fdnum].fileext;
            if(!ret) *error = 1;
            break;
        case TYPE_UNICODE:
            ret  = fgetss(fdnum, 0,    1, 0);
            if(!ret) *error = 1;
            break;
        default: {
            printf("\nError: invalid datatype %d\n", type);
            myexit(-1);
            break;
        }
    }
    *ret_num = retn;
    //if(!ISNUMTYPE(type) && !ret) *error = 1;  // bad, decrease a lot the performances
    return(ret);
}



void bytesread_eof(int fdnum, int len) {
    int     oldoff  = 0;

    if(!fdnum) {
        oldoff = get_var32(BytesRead_idx);
        add_var(BytesRead_idx, NULL, NULL, oldoff + len, sizeof(int));
        if(myftell(fdnum) >= myfilesize(fdnum)) {
        //if(myfeof(fdnum)) {   // feof doesn't work
            add_var(NotEOF_idx, NULL, NULL, 0, sizeof(int));
        }
    }
}



void post_fseek_actions(int fdnum, int diff_offset) {
    if(file_xor_size)   (*file_xor_pos)   += diff_offset;
    if(file_rot13_size) (*file_rot13_pos) += diff_offset;
    if(mex_default) bytesread_eof(fdnum, diff_offset);
}



void post_fread_actions(int fdnum, u8 *data, int len) {
    int     i;

    if(file_xor_size) {
        for(i = 0; i < len; i++) {
            data[i] ^= file_xor[(*file_xor_pos) % file_xor_size];
            (*file_xor_pos)++;
        }
    }
    if(file_rot13_size) {
        for(i = 0; i < len; i++) {
            data[i] += file_rot13[(*file_rot13_pos) % file_rot13_size];
            (*file_rot13_pos)++;
        }
    }
    if(mex_default) bytesread_eof(fdnum, len);
}



int myftell(int fdnum) {
    if(fdnum < 0) {
        return(memory_file[-fdnum].pos);
    }
    if(!filenumber[fdnum].fd) {
        printf("\nError: the specified file number (%d) has not been opened yet\n", fdnum);
        myexit(-1);
    }
    return(ftell(filenumber[fdnum].fd));
}



int myfeof(int fdnum) {
    memory_file_t   *memfile    = NULL;
    int     ret = 0;

    if(fdnum < 0) {
        memfile = &memory_file[-fdnum];
        if(memfile->pos >= memfile->size) {
            ret = 1;
        }
    } else {
        if(!filenumber[fdnum].fd) {
            printf("\nError: the specified file number (%d) has not been opened yet\n", fdnum);
            myexit(-1);
        }
        ret = feof(filenumber[fdnum].fd);
    }
    return(ret);
}



void myfseek(int fdnum, int offset, int type) {
    memory_file_t   *memfile    = NULL;
    int     oldoff,
            err = 0;

    oldoff = myftell(fdnum);
    if(fdnum < 0) {
        memfile = &memory_file[-fdnum];
        switch(type) {
            case SEEK_SET: memfile->pos = offset;                   break;
            case SEEK_CUR: memfile->pos += offset;                  break;
            case SEEK_END: memfile->pos = memfile->size + offset;   break;
            default: break;
        }
        if((memfile->pos < 0) || (memfile->pos > memfile->size)) {
            err = -1;
        }
    } else {
        if(!filenumber[fdnum].fd) {
            printf("\nError: the specified file number (%d) has not been opened yet\n", fdnum);
            myexit(-1);
        }
        err = fseek(filenumber[fdnum].fd, offset, type);
    }
    if(err) {
        printf("\nError: the specified offset can't be reached\n");
        myexit(-1);
    }
    post_fseek_actions(fdnum, (int)myftell(fdnum) - oldoff);
}



int myfr(int fdnum, u8 *data, int size) {
    memory_file_t   *memfile    = NULL;
    int     len,
            quit_if_diff    = 1;

    if(size < 0) {
        size = BUFFSZ;
        quit_if_diff = 0;
    }
    if(fdnum < 0) {
        memfile = &memory_file[-fdnum];
        if(!memfile->data) {
            printf("\nError: MEMORY_FILE has not been used in this script yet\n");
            myexit(-1);
        }
        len = size;
        if((memfile->pos + size) > memfile->size) {
            len = memfile->size - memfile->pos;
        }
        memcpy(data, memfile->data + memfile->pos, len);
        memfile->pos += len;
    } else {
        if(!filenumber[fdnum].fd) {
            printf("\nError: the specified file number (%d) has not been opened yet\n", fdnum);
            myexit(-1);
        }
        len = fread(data, 1, size, filenumber[fdnum].fd);
    }
    if((len != size) && quit_if_diff) {
        printf("\n"
            "Error: incomplete input file number %d, can't read %u bytes.\n"
            "       anyway don't worry, it's possible that the BMS script has been written\n"
            "       to exit in this way if it's reached the end of the archive so check it\n"
            "       or contact its author or verify that all the files have been extracted\n"
            "\n", fdnum, size - len);
        myexit(-1);
    }
    post_fread_actions(fdnum, data, len);
    return(len);
}



void myhelp(u8 *argv0) {
    printf("\n"
        "Usage: %s [options] <script.BMS> <input_archive/folder> <output_folder>\n"
        "\n"
        "Options:\n"
        "-l     list the files without extracting them, you can use . as output folder\n"
        "-f W   filter the files to extract using the W wildcard, example -f \"*.mp3\"\n"
        "-F W   as above but works only with the files in the input folder (if used)\n"
        "-o     if the output files already exist this option will overwrite them\n"
        "       automatically without asking the user's confirmation\n"
        "-v     verbose debug informations, useful for verifying possible errors\n"
        "-c     quick list of BMS commands and some notes about them and this tool\n"
        "-L F   dump the offset/size/name of the files inside the file F\n"
        "-x     use the hexadecimal notation in myitoa (debug)\n"
        "\n"
        "Examples:\n"
        "  quickbms c:\\zip.bms c:\\myfile.zip \"c:\\new folder\"\n"
        "  quickbms -l -f \"*.txt\" zip.bms myfile.zip .\n"
        "\n", argv0);
}



void quick_bms_list(void) {
    fputs("\n"
        "quick reference list of the BMS commands:\n"
        "\n"
        " CLog <filename> <offset> <compressed_size> <uncompressed_size> [file]\n"
        "    extract the file at give offset decompressing its content\n"
        "\n"
        " Do\n"
        " ...\n"
        " While <variable> <condition> <variable>\n"
        "    perform a loop which ends when the condition is no longer valid\n"
        "\n"
        " FindLoc <variable> <type> <string> [file] [ret_if_err]\n"
        "    if the string is found put its offset in the variable\n"
        "    by default if FindLoc doesn't find the string it terminates the script\n"
        "    while if ret_if_err is specified (for example -1 or \"\") it will be put in\n"
        "    variable instead of terminating\n"
        "\n"
        " For [variable] = [value] [To] [variable]\n"
        " ...\n"
        " Next [variable]\n"
        "    classical for(;;) loop, Next simply increments the value of the variable\n"
        "    the arguments are optionals for using this For like an endless loop and\n"
        "    To can be substituited with any condition like != == < <= > >= and so on\n"
        "\n"
        " Break\n"
        "    quit a loop (experimental)\n"
        "\n"
        " Get <variable> <type> [file]\n"
        "    read a number (8, 16, 32 bits) or a string\n"
        "\n"
        " GetDString <variable> <length> [file]\n"
        "    read a string of the specified length\n"
        "\n"
        " GoTo <offset> [file]\n"
        "    reach the specified offset, if it's negative it goes from the end\n"
        "\n"
        " IDString [file] <string>\n"
        "    check if the data in the file matches the given string\n"
        "\n"
        " Log <filename> <offset> <size> [file]\n"
        "    extract the file at the given offset with that size\n"
        "\n"
        " Math <variable> <operator> <variable>\n"
        "    perform a mathematical operation on the first variable, available op:\n"
        "    + * / - ^ & | % ! ~ << >> r (rot right) l (rot left) s (bit s) w (byte s)\n"
        "\n"
        " Open <folder> <filename> <file>\n"
        "    open a new file and assign the given file number\n"
        "\n"
        " SavePos <variable> [file]\n"
        "    save the current offset in the variable\n"
        "\n"
        " Set <variable> [type] <variable>\n"
        "    assign the content of the second variable to the first one, type ignored\n"
        "\n"
        " String <variable> <operator> <variable>\n"
        "    perform an append/removing/xor operation on the first variable\n"
        "\n"
        " CleanExit\n"
        "    terminate the extraction\n"
        "\n"
        " If <variable> <criterium> <variable>\n"
        " ...\n"
        " Else / Elif / Else If\n"
        " ...\n"
        " EndIf\n"
        "    classical if(...) { ... } else { ... }\n"
        "\n"
        " GetCT <variable> <type> <character> [file]\n"
        "    read a string (type is useless) delimited by the given character\n"
        "\n"
        " ComType <type>\n"
        "    specify the type of compression to use in CLog: zlib1, lzo, deflate, lzss,\n"
        "    lzx, gzip, explode, lzma, bz2, XMemDecompress LZX, some lzw variants, hex,\n"
        "    base64, ms lzx (the one used in cab and chm files)\n"
        "\n"
        " ReverseLong <variable>\n"
        "    invert the order/endianess of the variable\n"
        "\n"
        " Endian <type>\n"
        "    choose between little and big endian order of the read numbers\n"
        "\n"
        " FileXOR <string_of_numbers> [offset]\n"
        "    xor the read data with the sequence of numbers in the given string\n"
        "\n"
        " FileRot13 <string_of_numbers> [offset]\n"
        "    add/substract the read data with the sequence of numbers in the string\n"
        "\n"
        " Strlen <variable> <variable>\n"
        "    put the length of the second variable in the first one\n"
        "\n"
        " GetVarChr <variable> <variable> <offset> [type]\n"
        "    put the byte at the given offset of the second variable in the first one\n"
        "\n"
        " PutVarChr <variable> <offset> <variable> [type]\n"
        "    put the byte in the second variable in the first one at the given offset\n"
        "\n"
        " Padding <number> [file]\n"
        "    adjust the current offset of the file using the specified number (size of\n"
        "    padding), note that at the moment the padding is performed only when\n"
        "    this command is called and not automatically after each file reading\n"
        "\n"
        " Append\n"
        "    enable/disable the writing of the data at the end of the files with *Log\n"
        "\n"
        " Encryption <algorithm> <key> [ivec]\n"
        "    enable that type of decryption: aes, blowfish, des/3des, rc4, tea, xtea,\n"
        "    idea, xor, rot, charset\n"
        "\n"
        " Print \"message\"\n"
        "    display a message, you can display the content of the variables simply\n"
        "    specifying their name between '%' like: Print \"my offset is %OFFSET%\"\n"
        "\n"
        " GetArray <variable> <array_num> <index>\n"
        "    get the value stored at the index position of array_num\n"
        "\n"
        " PutArray <array_num> <index> <variable>\n"
        "    store the variable at the index position of array_num\n"
        "\n"
        " StartFunction NAME\n"
        " ...\n"
        " EndFunction\n"
        " CallFunction NAME\n"
        "    experimental functions for recursive archives\n"
        "\n"
        "NOTES:\n"
        "- a variable and a fixed number are the same thing internally in the tool\n"
        "  because all the data is handled as strings with the consequent pros\n"
        "  (incredibly versatile) and cons (slowness with some types of scripts)\n"
        "- everything is case insensitive (ABC is like abc) except the content of\n"
        "  strings and variables (excluded some operations like in String)\n"
        "- the [file] field is optional, if not specified it's 0 so the main file\n"
        "- also the final ';' char of the original BMS language is optional\n"
        "- example of <string_of_numbers>: \"0x123 123 456 -12 -0x7f\" or 0xff or \"\"\n"
        "- both hexadecimal (0x) and decimal numbers are supported, negatives included\n"
        "- all the mathematical operations are performed using signed 32 bit numbers\n"
        "- available types of data: long (32 bits), short (16), byte (8), string\n"
        "- all the fixed strings are handled in C syntax like \"\\x12\\x34\\\\hello\\\"bye\\0\"\n"
        "- do not use variable names which start with a number like 123MYVAR or -MYVAR\n"
        "- if you use the file MEMORY_FILE will be used a special memory buffer, create\n"
        "  it with CLog or Log and use it normally like any other file\n"
        "- is possible to use multiple memory files: MEMORY_FILE, MEMORY_FILE2,\n"
        "  MEMORY_FILE3, MEMORY_FILE4 and so on\n"
        "- use TEMPORARY_FILE for creating a file with this exact name also in -l mode\n"
        "\n"
        "informations about the original BMS scripting language and original examples:\n"
        "  http://wiki.xentax.com/index.php/BMS\n"
        "  http://multiex.xentax.com/complete_scripts.txt\n"
        "\n"
        "check the source code of this tool for the additional enhancements implemented\n"
        "by me (like support for xor, rot13, lzo, lzss, zlib/deflate and so on) or send\n"
        "me a mail because various features are not documented yet or just watch the\n"
        "examples provided on the project's homepage which cover ALL the enhancements:\n"
        "  http://aluigi.org/papers.htm#quickbms\n"
        "\n"
        "the tool supports also the \"multiex inifile\" commands in case of need.\n"
        "\n", stdout);
}



int calc_quickbms_version(u8 *version) {
    int     n,
            len,
            ret,
            seq;
    u8      *p;

    ret = 0;
    seq = 24;
    for(p = version; *p; p += len) {
        if(*p == '.') {
            seq -= 8;
            if(seq < 0) break;
            len = 1;
        } else if((*p >= 'a') && (*p <= 'z')) {
            ret += *p;
            len = 1;
        } else {
            n = readbase(p, 10, &len);
            if(len <= 0) break;
            ret += n << seq;
        }
    }
    return(ret);
}



void alloc_err(const char *fname, int line, const char *func) {
    printf("\n- error in %s line %d: %s()\n", fname, line, func);
    printf("Error: tentative of allocating -1 bytes\n");
    myexit(-1);
}



void std_err(const char *fname, int line, const char *func) {
    printf("\n- error in %s line %d: %s()\n", fname, line, func);
    perror("Error");
    myexit(-1);
}



void myexit(int ret) {
    if(!ret && quick_gui_exit) exit(ret);   // as below
#ifdef WIN32
    u8      ans[16];

    if(GetWindowLong(mywnd, GWL_WNDPROC)) {
        printf("\nPress RETURN to quit");
        fgets(ans, sizeof(ans), stdin);
    }
#endif
    exit(ret);  // as above
}


